unit IBBDEDatabase;

{ IBBDEDatabase

  by Craig Stuntz (cstuntz@vertexsoftware.com)

  Copyright 2001 by Vertex Systems Corporation.  Permission is granted to
  Borland to redistribute in accortance with the terms and conditions in the
  CodeCentral license agreement.

  Allows IBX and the BDE to share a single connection to an InterBase
  database.

  Unsupported freeware -- use at your own risk!

  Please do not email me support questions -- instead, post them to
  news://newsgroups.borland.com/borland.public.delphi.database.interbaseexpress

  To use:  Create a new package.  Add this file to the package.  Create a
  Register procedure either in this unit or the package to register this
  component.  Build the package.

  }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Db, IBDatabase, BDE, DBTables;

const
  BDEDATABASENOTASSIGNED = 'BDEDatabase property not assigned.';

type
  TIBBDEDatabase = class(TIBDatabase)
  private
    bBDEClosing, bClosing, bStreamedConnected: boolean;
    FBDEDatabase: TDatabase;
    procedure BDEHandleChanged(const Connected: boolean);
    function  GetNativeDBHandle(DBHandle: hDBIDb; var NativeHandle: longint): Word;
    procedure InitializeFields;
    procedure OnConnect(Sender: TObject; Connecting: Boolean);
    procedure SetBDEDatabase(const Value: TDatabase);
  protected
    procedure DoConnect; override;
    procedure DoDisconnect; override;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure SetConnected(Value: Boolean); override;
  published
    property    BDEDatabase: TDatabase read FBDEDatabase write SetBDEDatabase;
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
  end;

  TDatabaseCracker = class(TDatabase);

procedure Register;

implementation

uses
  IB, IBHeader;

procedure Register;
begin
  RegisterComponents('InterBase', [TIBBDEDatabase]);
end;

{ TIBBDEDatabase }

procedure TIBBDEDatabase.BDEHandleChanged(const Connected: boolean);
var
  hDB: TISC_DB_HANDLE;
begin
  if Connected then begin
    GetNativeDBHandle(FBDEDatabase.Handle, Longint(hDB));
  end else begin
    hDB := nil;
  end;
  if hDB <> Handle then begin
    SetHandle(hDB);
  end;
end;

constructor TIBBDEDatabase.Create(AOwner: TComponent);
begin
  InitializeFields;
  inherited;
end;

destructor TIBBDEDatabase.Destroy;
begin
  // remove Self from TDatabase's list of registered Clients.
  if Assigned(BDEDatabase) then begin
    BDEDatabase := nil;
  end; // if
  inherited;
end;

procedure TIBBDEDatabase.DoConnect;
begin
  if (not Connected) and (Assigned(FBDEDatabase)) then begin
    if (csReading in ComponentState) then begin
      // defer opening until streaming finished
      bStreamedConnected := TRUE;
    end else begin
      FBDEDatabase.Connected := TRUE;
    end; // if
  end;// if
  bClosing := FALSE;
end;

procedure TIBBDEDatabase.DoDisconnect;
begin
  { When setting connected FALSE, we need to close the BDE TDatabase.  This
    is because we want Self and the BDEDatabases to be synchronized -- when one
    is open, so is the other; when one is closed, so is the other.  The
    notification mechanism makes this happen.  The trouble comes when closing
    the connection.  This routine will attempt to Close the BDE TDatabase, which
    will send a closing notification back to this routine, which will see that
    the BDE Database hasn't closed yet, and will attempt to Close the BDE
    TDatabase, and so on forever.  The solution is to set a flag to make sure
    the routine only runs once. }
  if (not bClosing) and Assigned (BDEDatabase) then begin
    bClosing := TRUE;
    try
      if not bBDEClosing then begin
        BDEDatabase.Connected := FALSE;
      end; // if
    finally
      bClosing := FALSE;
    end;
  end;
end;

function TIBBDEDatabase.GetNativeDBHandle(DBHandle: hDBIDb;
  var NativeHandle: longint): Word;
begin
  Result := 0;
  // Get the native handle to the database...
  Check(DbiGetProp(hDBIObj(DBHandle), dbNATIVEHNDL, @NativeHandle,
    sizeof(NativeHandle), Result));
end;

procedure TIBBDEDatabase.InitializeFields;
begin
  bStreamedConnected := FALSE;
  bClosing := FALSE;
  bBDEClosing := FALSE;
end;

procedure TIBBDEDatabase.Loaded;
begin
  inherited;
  Connected := bStreamedConnected;
end;

procedure TIBBDEDatabase.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if AComponent = BDEDatabase then begin
      FBDEDatabase := nil;
      SetHandle(nil);
    end; // if
  end;
end;

procedure TIBBDEDatabase.OnConnect(Sender: TObject; Connecting: Boolean);
begin
  if Sender = FBDEDatabase then begin
    if Connecting then begin
      BDEHandleChanged(Connecting);
    end else begin
      if not bBDEClosing then begin
        bBDEClosing := TRUE;
        try
          BDEHandleChanged(Connecting);
        finally
          bBDEClosing := FALSE;
        end; // try-finally
      end; // if
    end; // if
  end; // if
end;

procedure TIBBDEDatabase.SetBDEDatabase(const Value: TDatabase);
begin
  { Tell previous database not to send us notifications anymore.
    Do this only if we're being set to a different database and not
    to nil.  It's dangerous to UnRegister if Value is nil because
    this can be called while the TDatabase is looping though its clients.
    TDatabase doesn't check each iteration to see if the client is still
    in the list, so unregistering in the middle of the loop causes a
    List index out of bounds error. }
  if Assigned(FBDEDatabase) and Assigned (Value) then begin
    TDatabaseCracker(FBDEDatabase).UnRegisterClient(Self);
  end; // if
  // set new database
  FBDEDatabase := Value;
  if Assigned(Value) then begin
    // ask for notifications
    TDatabaseCracker(FBDEDatabase).RegisterClient(Self, OnConnect);
    FBDEDatabase.FreeNotification(Self);
    BDEHandleChanged(FBDEDatabase.Connected);
  end else begin
    BDEHandleChanged(FALSE);
  end;
end;

procedure TIBBDEDatabase.SetConnected(Value: Boolean);
begin
  if Value and (not Connected) and not Assigned(FBDEDatabase) and
    // Self may load before BDEDatabase, so ignore this error while loading
    (not ( csReading in ComponentState)) then begin
    raise EIBError.Create(BDEDATABASENOTASSIGNED);
  end; // if
  // this actually opens the DB
  inherited;
end;

end.
