{+--------------------------------------------------------------------------+
 | Component: TIBDatabase
 | Author: Robert J. Love
 | Company: Lovedata Software
 | Copyright 1996, all rights reserved.
 | Description: Interbase Database Component
 | Version: 0.5a
 | Modification History: NONE
 | Updates may be obtained at http://www.xmission.com/~uldata/ib
 |
 | This Code May not be modified and sold for Profit
 | Any Modification to this Code that is Redistributed must
 | include source code and must be freely available to the public and
 | must include the above Information
 +--------------------------------------------------------------------------+}
unit IBdb;  { TIBDatabase component. }

interface

uses
  WinTypes,
  WinProcs,
  SysUtils,
  Messages,  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  Menus,
  IBProcs,
  IBbase;
type
  TTransactionEvent = procedure of object;

type
  TConnectEvent = procedure of object;
  TDisConnectEvent = procedure of object;
  TIBDatabase = class(TIBBase)
  private
    FTran : Boolean; {Is there an Active Transaction?}
    FOnBeforeStartTransaction : TTransactionEvent;
    FOnAfterStartTransaction : TTransactionEvent;
    FOnBeforeCommitTransaction : TTransactionEvent;
    FOnAfterCommitTransaction : TTransactionEvent;
    FOnBeforeRollbackTransaction : TTransactionEvent;
    FOnAfterRollbackTransaction : TTransactionEvent;
    { Private declarations }
    FUsername : String;
    FPassword : string;
    FDatabase : String;
    FConnected : Boolean;
    FParams : TStringList;
    FOnBeforeConnect : TConnectEvent;
    FOnAfterConnect : TConnectEvent;
    FOnBeforeDisConnect : TDisConnectEvent;
    FOnAfterDisConnect : TDisConnectEvent;
    function GetConnected : Boolean;
    procedure SetParams(newValue : TStringList);
  protected
    { Protected declarations }
    Buffer: array[0..1023] of char;
    BufPtr: integer;
    procedure initDPB;
    procedure BuildDPB( item: byte; contents: string);
    { Event triggers: }
    procedure TriggerBeforeConnectEvent; virtual;
    procedure TriggerAfterConnectEvent; virtual;
    procedure TriggerBeforeDisConnectEvent; virtual;
    procedure TriggerAfterDisConnectEvent; virtual;
    procedure TriggerBeforeStartTransactionEvent; virtual;
    procedure TriggerAfterStartTransactionEvent; virtual;
    procedure TriggerBeforeCommitTransactionEvent; virtual;
    procedure TriggerAfterCommitTransactionEvent; virtual;
    procedure TriggerBeforeRollbackTransactionEvent; virtual;
    procedure TriggerAfterRollbackTransactionEvent; virtual;
  public
    { Public declarations }
    dbHandle : isc_db_Handle;
    trHandle : isc_tr_handle;
    teb : isc_teb;
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Connect;
    procedure DisConnect;
    procedure StartTransaction;
    procedure Commit;
    procedure Rollback;
  published
    { Published properties and events }
    property Username : String read FUsername write FUsername;
    property Password : string read FPassword write FPassword;
    property DatabaseName : string read FDatabase write fdatabase;
    property ActiveTransaction : boolean read fTran; //ReadOnly
    property Connected : Boolean read GetConnected; //ReadOnly
    property Params : TStringList read FParams write SetParams;
    property OnBeforeConnect : TConnectEvent read FOnBeforeConnect write FOnBeforeConnect;
    property OnAfterConnect : TConnectEvent read FOnAfterConnect write FOnAfterConnect;
    property OnBeforeDisConnect : TDisConnectEvent read FOnBeforeDisConnect write FOnBeforeDisConnect;
    property OnAfterDisConnect : TDisConnectEvent read FOnAfterDisConnect write FOnAfterDisConnect;
    Property OnIBError; //Publish Inherited Error Event
    property OnBeforeStartTransaction : TTransactionEvent read FOnBeforeStartTransaction write FOnBeforeStartTransaction;
    property OnAfterStartTransaction : TTransactionEvent read FOnAfterStartTransaction write FOnAfterStartTransaction;
    property OnBeforeCommitTransaction : TTransactionEvent read FOnBeforeCommitTransaction write FOnBeforeCommitTransaction;
    property OnAfterCommitTransaction : TTransactionEvent read FOnAfterCommitTransaction write FOnAfterCommitTransaction;
    property OnBeforeRollbackTransaction : TTransactionEvent read FOnBeforeRollbackTransaction write 
      FOnBeforeRollbackTransaction;
    property OnAfterRollbackTransaction : TTransactionEvent read FOnAfterRollbackTransaction write FOnAfterRollbackTransaction;
  end;  { TIBDatabase }

procedure Register;

implementation

function TIBDatabase.GetConnected : Boolean;
{ Returns the value of data member FConnnected. }
begin
  result := FConnected;
end;  { GetConnnected }

procedure TIBDatabase.SetParams(newValue : TStringList);
{ Sets data member FParams to newValue. }
begin
  FParams.Assign(newValue);
end;  { SetParams }

procedure TIBDatabase.InitDPB;
begin
  //InitBuffer
  FillChar(Buffer,sizeof(Buffer),#0);
  BufPtr := 0;
  //InitDPB
  Buffer[0] := char(isc_dpb_version1);
  inc(BufPtr);
end;

procedure TIBDatabase.BuildDPB( item: byte; contents: string);
begin
  Buffer[BufPtr] := char(item);
  inc(BufPtr);
  Buffer[BufPtr] := char(length(contents));
  inc(BufPtr);
  StrPCopy(@Buffer[ BufPtr],Contents);
  inc(BufPtr,length(Contents));
end;


procedure TIBDatabase.Connect;  { public }
 var
  DBName: array[0..255] of char;
  status: status_vector;
  errCode: isc_status;
begin
   if FConnected then Disconnect;
   TriggerBeforeConnectEvent;
   StrPCopy( DBName, FDatabase);
   DBHandle := nil;
   InitDPB;
   BuildDPB( isc_dpb_user_name, FUsername);
   BuildDPB( isc_dpb_password, FPassword);
   errCode := isc_attach_database( @status, Length(FDatabase), @DBName, @DBHandle,
                                  BufPtr, @Buffer);
   if errCode <> 0 then HandleIBErrors( @status)
                   else FConnected := True;
   TriggerAfterConnectEvent;
end;  { Connect }

procedure TIBDatabase.DisConnect;  { public }
var
 status : status_vector;
 errCode : isc_status;
begin
      triggerBeforeDisConnectEvent;
      errCode := isc_detach_database( @status, @DbHandle);
      if errCode <> 0 then HandleIBErrors( @status);
      FConnected := False;
      triggerAfterDisConnectEvent;
end;  { DisConnect }

{ Event triggers: }
procedure TIBDatabase.TriggerBeforeConnectEvent;
{ Triggers the OnBeforeConnect event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnBeforeConnect)) then
    FOnBeforeConnect;
end;  { TriggerBeforeConnectEvent }

procedure TIBDatabase.TriggerAfterConnectEvent;
{ Triggers the OnAfterConnect event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnAfterConnect)) then
    FOnAfterConnect;
end;  { TriggerAfterConnectEvent }

procedure TIBDatabase.TriggerBeforeDisConnectEvent;
{ Triggers the OnBeforeDisConnect event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnBeforeDisConnect)) then
    FOnBeforeDisConnect;
end;  { TriggerBeforeDisConnectEvent }

procedure TIBDatabase.TriggerAfterDisConnectEvent;
{ Triggers the OnAfterDisConnect event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnAfterDisConnect)) then
    FOnAfterDisConnect;
end;  { TriggerAfterDisConnectEvent }

procedure TIBDatabase.StartTransaction;  { public }
begin
  TriggerBeforeStartTransactionEvent;
  trHandle := nil;
  teb.db_ptr := @DBHandle;;
  teb.tpb_len := 0;
  teb.tpb_ptr := nil;
  errcode := isc_start_multiple( @status, @trHandle, 1, @teb);
  if errcode <> 0 then HandleIBErrors( @status);
  FTran := True;
  TriggerAfterStartTransactionEvent;
end;  { StartTransaction }

procedure TIBDatabase.Commit;  { public }
begin
    TriggerBeforeCommitTransactionEvent;
    errcode := isc_commit_transaction( @status, @trHandle);
    if errcode <> 0 then HandleIBErrors( @status);
    Ftran := False;
    TriggerAfterCommitTransactionEvent;
end;  { Commit }

procedure TIBDatabase.Rollback;  { public }
begin
    TriggerBeforeRollbackTransactionEvent;
    errcode := isc_rollback_transaction( @status, @trHandle);
    if errcode <> 0 then HandleIBErrors( @status);
    FTran := False;
    TriggerAfterRollbackTransactionEvent;
end;  { Rollback }

procedure TIBDatabase.TriggerBeforeStartTransactionEvent;
{ Triggers the OnBeforeStartTransaction event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnBeforeStartTransaction)) then
    FOnBeforeStartTransaction;
end;  { TriggerBeforeStartTransactionEvent }

procedure TIBDatabase.TriggerAfterStartTransactionEvent;
{ Triggers the OnAfterStartTransaction event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnAfterStartTransaction)) then
    FOnAfterStartTransaction;
end;  { TriggerAfterStartTransactionEvent }

procedure TIBDatabase.TriggerBeforeCommitTransactionEvent;
{ Triggers the OnBeforeCommitTransaction event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnBeforeCommitTransaction)) then
    FOnBeforeCommitTransaction;
end;  { TriggerBeforeCommitTransactionEvent }

procedure TIBDatabase.TriggerAfterCommitTransactionEvent;
{ Triggers the OnAfterCommitTransaction event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnAfterCommitTransaction)) then
    FOnAfterCommitTransaction;
end;  { TriggerAfterCommitTransactionEvent }

procedure TIBDatabase.TriggerBeforeRollbackTransactionEvent;
{ Triggers the OnBeforeRollbackTransaction event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnBeforeRollbackTransaction)) then
    FOnBeforeRollbackTransaction;
end;  { TriggerBeforeRollbackTransactionEvent }

procedure TIBDatabase.TriggerAfterRollbackTransactionEvent;
{ Triggers the OnAfterRollbackTransaction event. This is a virtual method (descendants of this component can override it). }
begin
  if (assigned(FOnAfterRollbackTransaction)) then
    FOnAfterRollbackTransaction;
end;  { TriggerAfterRollbackTransactionEvent }

destructor TIBDatabase.Destroy;
begin
  { Free member variables: }
  FParams.Free;
  inherited Destroy;
end;  { Destroy }

constructor TIBDatabase.Create(AOwner : TComponent);
{ Creates an object of type TIBDatabase, and initializes properties. }
begin
  inherited Create(AOwner);
  { Create property fields (that are objects): }
  FParams := TStringList.Create;
end;  { Create }

procedure Register;
begin
  RegisterComponents('Interbase', [TIBDatabase]);
end;  { Register }

end.

