{******************************************************************************}
{                                                                              }
{           ReportBuilder Data Access Developement Environment (DADE)          }
{                                                                              }
{             Copyright (c) 1996-1999 Digital Metaphors Corporation            }
{                                                                              }
{                This unit is a modified version of daODBC98.pas               }
{                                                                              }
{                  Modified by Adrian Billingham adrian@softly.demon.co.uk     }
{                                                                              }
{******************************************************************************}

unit daFIB;

interface

uses Classes, SysUtils, Forms, ExtCtrls,  DB,
     ppClass, ppDBPipe, ppDB, ppClasUt,pptypes,
     daDB, daQuery, daDataVw, daDatPrv,
     ibase,FIBQuery,FIBDataset, FIBDatabase;

type

  TIBTable = class (TFIBDataset)
    public TableName : String;
  end;

  { TdaChildIBQuery }
  TdaChildIBQuery = class(TIBTable)
    public
      function HasParent: Boolean; override;
    end;  {class, TdaChildIBQuery}

  { TdaChildIBTable }
  TdaChildIBTable = class(TIBTable)
    public
      function HasParent: Boolean; override;
    end;  {class, TdaChildIBTable}

  { TdaChildIBStoredProc }
  TdaChildIBStoredProc = class(TFIBQuery)
    public
      function HasParent: Boolean; override;
    end;  {class, TdaChildIBStoredProc}


  { TdaIBSession }
  TdaIBSession = class(TdaSession)
    private

    public

      class function ClassDescription: String; override;
      class function DataSetClass: TdaDataSetClass; override;
      function GetAliasDriverName(const aAlias: String): String; override;
      procedure GetDatabaseNames(aList: TStrings); override;
      procedure GetTableNames(const aDatabaseName: String; aList: TStrings); override;

  end; {class, TdaIBSession}

  { TdaIBDataSet }
  TdaIBDataSet = class(TdaDataSet)
    private
      FTable: TIBTAble;
      function GetTable: TIBTable;

    protected
      procedure BuildFieldList; override;
      function  GetActive: Boolean; override;
      procedure SetActive(Value: Boolean); override;
      procedure SetDatabaseName(const aDatabaseName: String); override;
      procedure SetDataName(const aDataName: String); override;

      property Table: TIBTable read GetTable;

    public
      constructor Create(aOwner: TComponent); override;
      destructor Destroy; override;

      class function ClassDescription: String; override;

      procedure GetFieldNamesForSQL(aList: TStrings; aSQL: TStrings); override;
      procedure GetFieldsForSQL(aList: TList; aSQL: TStrings); override;

  end; {class, TdaIBDataSet}


  { TdaIBQueryDataView }
  TdaIBQueryDataView = class(TdaQueryDataView)
    private
      FDataSource: TppChildDataSource;
      FQuery: TdaChildIBQuery;
      FTimer: TTimer;

      procedure AfterLoadededEvent(Sender: TObject);

    protected
      procedure Loaded; override;
      procedure UpdateQueryObject;

      procedure SetReport(aReport: TppCustomReport); override;
      procedure SetActive(Value: Boolean); override;

      procedure SQLChanged; override;

    public
      constructor Create(aOwner: TComponent); override;
      destructor Destroy; override;

      class function PreviewFormClass: TFormClass; override;
      class function SessionClass: TClass; override;

      procedure Init; override;

    published
      property DataSource: TppChildDataSource read FDataSource;

  end; {class, TdaIBQueryDataView}

  {global functions to access default IB session and database}
  function daGetDefaultIBDatabase: TFIBDatabase;

  {utility routines}
  procedure daGetIBDatabaseNames(aList: TStrings);
  function daGetIBDatabaseForName(aDatabaseName: String): TFIBDatabase;

  {Delphi design time registration}

procedure Register;

implementation

const
  cDefaultDatabase = 'DefaultIBDatabase';

var
  IBDatabase: TFIBDatabase;

{******************************************************************************
 *
 ** R E G I S T E R
 *
{******************************************************************************}

procedure Register;
begin

  RegisterNoIcon([TdaChildIBQuery, TdaChildIBTable, TdaChildIBStoredProc]);

  {IB DataViews}
  RegisterNoIcon([TdaIBQueryDataView]);
end;

{******************************************************************************
 *
 ** C H I L D   B D E  D A T A   A C C E S S   C O M P O N  E N T S
 *
{******************************************************************************}

{------------------------------------------------------------------------------}
{ TdaChildIBQuery.HasParent }

function TdaChildIBQuery.HasParent: Boolean;
begin
  Result := True;
end; {function, HasParent}

{------------------------------------------------------------------------------}
{ TdaChildIBTable.HasParent }

function TdaChildIBTable.HasParent: Boolean;
begin
  Result := True;
end; {function, HasParent}


{------------------------------------------------------------------------------}
 {TdaChildIBStoredProc.HasParent }

function TdaChildIBStoredProc.HasParent: Boolean;
begin
  Result := True;
end; {function, HasParent}

{******************************************************************************
 *
 ** B D E   S E S S I O N
 *
{******************************************************************************}

{------------------------------------------------------------------------------}
{ TdaIBSession.ClassDescription }

class function TdaIBSession.ClassDescription: String;
begin
  Result := 'FIBsession';
end; {class function, ClassDescription}


{------------------------------------------------------------------------------}
{ TdaIBSession.DataSetClass }

class function TdaIBSession.DataSetClass: TdaDataSetClass;
begin
  Result := TdaIBDataSet;
end; {class function, DataSetClass}

{------------------------------------------------------------------------------}
{ TdaIBSession.GetTableNames }

procedure TdaIBSession.GetTableNames(const aDatabaseName: String; aList: TStrings);
var
  lDatabase: TFibDatabase;
  // to fetch table nam4es we must query the system tables
  qry : TFIBQuery;

begin
  {get the database}
  lDatabase := daGetIBDatabaseForName(aDatabaseName);
  {connection must be active to get table names}
  if not lDatabase.Connected then
  begin
    lDatabase.Connected := True;
    ldatabase.Defaulttransaction.Active:=True;
  end;

  if lDatabase.Connected then
  begin
    qry:=TFibQuery.Create(nil);
    qry.Database:=ldatabase;
    qry.transaction:=ldatabase.Defaulttransaction;
    qry.sql.add('select RDB$RELATION_NAME as NAME ');
    qry.sql.add('from RDB$relations');
    qry.sql.add('where RDB$Relation_name not like ''RDB$%''');
    qry.sql.add('order by RDB$RELATION_NAME');

    alist.clear;
    // Put the table names in l

    with qry do
    begin
      execQuery;
      while not eof do
      begin
        alist.add(trim(FieldByname['NAME'].AsString));
        next;
      end;
    end;
    //assign to the correct list
    qry.free;
  end;
end; {procedure, GetTableNames}

{------------------------------------------------------------------------------}
{ TdaIBSession.GetDatabaseNames }

procedure TdaIBSession.GetDatabaseNames(aList: TStrings);
begin
  {call utility routine to get list of database names}
  {call utility routine to get list of database names}
  daGetIBDatabaseNames(aList);
end; {procedure, GetDatabaseNames}

{------------------------------------------------------------------------------}
{ TdaIBSession.GetAliasDriverName }

function TdaIBSession.GetAliasDriverName(const aAlias: String): String;
begin
  {return the driver name here to enable the Query Designer to generate server
   specifiC SQL: For example, generate a Left Join for Oracle}
  Result := '';
end; {procedure, GetAliasDriverName}

{******************************************************************************
 *
 ** B D E   D A T A S E T
 *
{******************************************************************************}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.Create }

constructor TdaIBDataSet.Create(aOwner: TComponent);
begin

  inherited Create(aOwner);

  FTable := nil;

end; {constructor, Create}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.Destroy }

destructor TdaIBDataSet.Destroy;
begin

  FTable.Free;

  inherited Destroy;

end; {destructor, Destroy}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.ClassDescription }

class function TdaIBDataSet.ClassDescription: String;
begin
  Result := 'FIBDataSet';
end; {class function, ClassDescription}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.GetActive }

function TdaIBDataSet.GetActive: Boolean;
begin
  Result := Table.Active;
end; {function, GetActive}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.SetActive }

procedure TdaIBDataSet.SetActive(Value: Boolean);
begin
  Table.Active := Value;
end; {procedure, SetActive}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.GetTable }

function TdaIBDataSet.GetTable: TIBTable;
begin
  {create IB table, if needed}
  if (FTable = nil) then
    FTable := TIBTable.Create(Self);

  Result := FTable;

end; {procedure, GetQuerty}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.SetDatabaseName }

procedure TdaIBDataSet.SetDatabaseName(const aDatabaseName: String);
begin

  inherited SetDatabaseName(aDatabaseName);

  {table cannot be active to set database property}
  if Table.Active then
    Table.Active := False;

  {get IB database for name}
  with Table do
  begin
    Database:=daGetIBDatabaseForName(aDatabaseName);
    Transaction:=database.DefaultTransaction;
  end;

end; {procedure, SetDatabaseName}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.SetDataName }

procedure TdaIBDataSet.SetDataName(const aDataName: String);
begin

  inherited SetDataName(aDataName);

  {table cannot be active to set table name property}
  if Table.Active then
    Table.Active := False;

  Table.TableName:=ADataName;
  With Table.SelectSQL do
  begin
    clear;
    add('select * from '+aDataName);
  end;
end; {procedure, SetDataName}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.BuildFieldList }

procedure TdaIBDataSet.BuildFieldList;
var
  liIndex: Integer;
  lIBField: TField;
  lField: TppField;
begin

  inherited BuildFieldList;

  {set table to active}
  if not(Table.Active) then
    try
      Table.Open;
    except
      on E:Exception do
        Exit;
    end; {try, except}

  {create TppField objects for each field in the table}
  for liIndex := 0 to Table.FieldCount - 1 do
    begin
      lIBField := Table.Fields[liIndex];

      lField := TppField.Create(nil);

      lField.TableName    := table.Tablename;
      lField.FieldName    := lIBField.FieldName;
      lField.DataType     := ppConvertFieldType(lIBField.DataType);

      AddField(lField);
    end;
end; {function, BuildFieldList}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.GetFieldNamesForSQL }

procedure TdaIBDataSet.GetFieldNamesForSQL(aList: TStrings; aSQL: TStrings);
var
  lQuery: TFIBDataSet;
  liIndex: Integer;
begin
  aList.Clear;

  lQuery := TFIBdataset.Create(Self);

  With lQuery do
  begin
    Database := daGetDefaultIBdatabase;
    Transaction:=database.DefaultTransaction;
    SelectSQL:=aSQL;
    Active:=True;
  end;

  for liIndex := 0 to lQuery.FieldCount - 1 do
      aList.Add(lquery.Fields[LiIndex].FieldName);
  lQuery.close;
  lQuery.Free;
end; {procedure, GetFieldNamesForSQL}

{------------------------------------------------------------------------------}
{ TdaIBDataSet.GetFieldsForSQL }

procedure TdaIBDataSet.GetFieldsForSQL(aList: TList; aSQL: TStrings);
var
  lQuery: TFIBdataset;
  lIBField : TField;
  lField: TppField;
  liIndex: Integer;
begin
  aList.Clear;

  {create a temporary IB query}
  lQuery := TFIBDataSet.Create(Self);

  {assign databae and SQL properties}
  With lQuery do
  begin
    Database := daGetIBDatabaseForName(DatabaseName);
    Transaction:=database.DefaultTransaction;
    SelectSQL:=aSQL;
  end;

  {set query to active}
  try
    lQuery.Active:=true;
  except
    on E:Exception do
      Exit;
  end; {try, except}

  {create a TppField object for each field in the query}

  for liIndex := 0 to lquery.Fieldcount-1 do
    begin
      lIBField := lQuery.Fields[liindex];  // lQuery.Field[liIndex];

      lField := TppField.Create(nil);
      lField.FieldName    := lIBField.FieldName;
      lField.DataType     := ppConvertFieldType(lIBField.DataType);
      aList.Add(lField);
    end;
  lQuery.Free;

end; {procedure, GetFieldsForSQL}

{******************************************************************************
 *
 ** B D E  Q U E R Y   D A T A V I E W
 *
{******************************************************************************}

{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.Create }

constructor TdaIBQueryDataView.Create(aOwner: TComponent);
begin

  inherited Create(aOwner);

  {notes: 1. must use ChildQuery, ChildDataSource, ChildPipeline etc.
          2. use Self as owner for Query, DataSource etc.
          3. do NOT assign a Name }

  FQuery := TdaChildIBQuery.Create(Self);

  FDataSource := TppChildDataSource.Create(Self);
  FDataSource.DataSet := FQuery;

end; {constructor, Create}

{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.Destroy }

destructor TdaIBQueryDataView.Destroy;
begin
  FDataSource.Free;
  FQuery.Free;

  inherited Destroy;

end; {destructor, Destroy}

{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.Loaded }

procedure TdaIBQueryDataView.Loaded;
begin

  inherited Loaded;

  if DataPipelineCount = 0 then Exit;

  {need to reconnect here}
  TppDBPipeline(DataPipelines[0]).DataSource := FDataSource;

  UpdateQueryObject;

  FTimer := TTimer.Create(Self);
  FTimer.OnTimer := AfterLoadededEvent;

end; {procedure, Loaded}


{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.AfterLoadededEvent }

procedure TdaIBQueryDataView.AfterLoadededEvent(Sender: TObject);
begin

  SetActive(True);

  FTimer.Free;
  FTimer := nil;

end; {procedure, AfterLoadededEvent}

{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.PreviewFormClass }

class function TdaIBQueryDataView.PreviewFormClass: TFormClass;
begin
  Result := TFormClass(GetClass('TdaPreviewDataDialog'));
end; {class function, PreviewFormClass}

{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.SessionClass }

class function TdaIBQueryDataView.SessionClass: TClass;
begin
  Result := TdaIBSession;
end; {class function, SessionClass}

{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.Init }

procedure TdaIBQueryDataView.Init;
var
  lDataPipeline: TppChildDBPipeline;

begin

  inherited Init;

  if DataPipelineCount > 0 then Exit;

  {note: DataView's owner must own the DataPipeline }
  lDataPipeline := TppChildDBPipeline(ppComponentCreate(Self, TppChildDBPipeline));
  lDataPipeline.DataSource := FDataSource;

  lDataPipeline.AutoCreateFields := False;

  {add DataPipeline to the dataview }
  lDataPipeline.DataView := Self;

end; {procedure, Init}

{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.SetReport }

procedure TdaIBQueryDataView.SetReport(aReport: TppCustomReport);
begin
  inherited SetReport(aReport);
  if (Report = nil) or (DataPipelineCount = 0) then Exit;
  Report.DataPipeline := DataPipelines[0];
end; {procedure, SetReport}



{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.SetActive }

procedure TdaIBQueryDataView.SetActive(Value: Boolean);
begin
  inherited SetActive(Value);
  if (DataPipelineCount = 0) then Exit;
  if DataPipelines[0].Active <> Value then
    DataPipelines[0].Open;
end; {procedure, SetActive}

{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.SQLChanged }

procedure TdaIBQueryDataView.SQLChanged;
begin
  UpdateQueryObject;
  inherited SQLChanged;
end; {procedure, WizardCompleted}

{------------------------------------------------------------------------------}
{ TdaIBQueryDataView.UpdateQueryObject }

procedure TdaIBQueryDataView.UpdateQueryObject;
begin

  if FQuery.Active then
    FQuery.Close;

   FQuery.Database:=daGetIBDatabaseForName(SQL.DatabaseName);
   FQuery.Transaction:=FQuery.Database.DefaultTransaction;

   FQuery.SelectSQL:=SQL.SQLText;

end; {procedure, UpdateQueryObject}

{******************************************************************************
 *
 ** P R O C E D U R E S   A N D   F U N C T I O N S
 *
{******************************************************************************}

{------------------------------------------------------------------------------}

procedure daGetIBDatabaseNames(aList: TStrings);

{Currently this procedure only fetches the default database name.           }
{Eventually functionality will be added to get all database names.          }
begin
  aList.Clear;
  aList.Add(daGetDefaultIBDatabase.dbname);
end; {procedure, daGetIBDatabaseNames}


{ daGetDefaultIBDatabase }

function daGetDefaultIBDatabase: TFIBDatabase;
begin
  {create the default IB database, if needed}
  if (IBDatabase = nil) then
    begin
      {create default IB database}
      IBDatabase := TFibdatabase.Create(nil);
      // Create a transaction
      IBdatabase.DBname:='d:\dev\windows accounts\data\winacc.gdb';
      IBdatabase.dbparams.add('user_name=SYSDBA');
      IBdatabase.dbparams.add('password=masterkey');
      IBDatabase.DefaultTransaction:=TFibTransaction.Create(nil);
    end;
   Result := IBDatabase;
end; {function, daGetDefaultIBDatabase}

{------------------------------------------------------------------------------}
{ daGetIBDatabaseForName }

function daGetIBDatabaseForName(aDatabaseName: String): TFIBdatabase;
begin
  Result:=daGetDefaultIBdatabase;
end; {function, daGetIBDatabaseForName}

initialization
  {register the IB descendant classes}
  RegisterClasses([TdaChildIBQuery, TdaChildIBTable, TdaChildIBStoredProc]);

  {register DADE descendant session, dataset, dataview}
  daRegisterSession(TdaIBSession);
  daRegisterDataSet(TdaIBDataSet);
  daRegisterDataView(TdaIBQueryDataView);

  {initialize internal reference variables}
  IBDatabase := nil;
finalization

  {free the default database object}
  IBDatabase.Free;

  {unregister the IB descendant classes}
  UnRegisterClasses([TdaChildIBQuery, TdaChildIBTable, TdaChildIBStoredProc]);

  {unregister DADE descendant the session, dataset, dataview}
  daUnRegisterSession(TdaIBSession);
  daUnRegisterDataSet(TdaIBDataSet);
  daUnRegisterDataView(TdaIBQueryDataView);
end.
