{********************************************************}
{                                                        }
{       Borland Deplphi                                  }
{       InterBase EventAlerter components                }
{       Copyright (c) 1995,98 Inprise Corporation        }
{                                                        }
{       Written by:                                      }
{         James Thorpe                                   }
{         CSA Australasia                                }
{         Compuserve: 100035,2064                        }
{         Internet:   csa@csaa.com.au                    }
{                                                        }
{       Adapted for FreeIBComponents                     }
{         12 Nov 1998 by Andre N Belokon                 }
{         support@softlab.od.ua                          }
{         http://softlab.od.ua                           }
{                                                        }
{********************************************************}

unit FIBEvent;

interface

uses
  SysUtils, Windows, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DB, IBase, ib_externals, FIB, FIBDatabase;

const
  MaxEvents = 15;
  EventLength = 64;

type

  TEventAlert = procedure( Sender: TObject; EventName: string; EventCount: longint;
                           var CancelAlerts: Boolean) of object;

  TEventBuffer = array[ 0..MaxEvents-1, 0..EventLength-1] of char;

  TFIBEventAlerter = class(TComponent)
  private
    FDatabase: TFIBDatabase;
    FEvents: TStrings;
    FOnEventAlert: TEventAlert;
    FQueued: Boolean;
    FRegistered: Boolean;
    Buffer: TEventBuffer;
    Changing: Boolean;
    CS: TRTLCriticalSection;
    EventBuffer: PChar;
    EventBufferLen: integer;
    EventID: Long;
    ProcessingEvents: Boolean;
    RegisteredState: Boolean;
    ResultBuffer: PChar;
    procedure DoQueueEvents;
    procedure EventChange( sender: TObject);
    procedure UpdateResultBuffer( length: short; updated: PChar);
    procedure ValidateDatabase(Database: TFIBDatabase);
  protected
    procedure HandleEvent;
    procedure Loaded; override;
    procedure Notification( AComponent: TComponent; Operation: TOperation); override;
    procedure SetEvents( value: TStrings);
    procedure SetDatabase( value: TFIBDatabase);
    procedure SetRegistered( value: boolean);
  public
    constructor Create( AOwner: TComponent); override;
    destructor Destroy; override;
    procedure CancelEvents;
    procedure QueueEvents;
    procedure RegisterEvents;
    procedure UnRegisterEvents;
    property  Queued: Boolean read FQueued;
  published
    property Database: TFIBDatabase read FDatabase write SetDatabase;
    property Events: TStrings read FEvents write SetEvents;
    property Registered: Boolean read FRegistered write SetRegistered;
    property OnEventAlert: TEventAlert read FOnEventAlert write FOnEventAlert;
  end;

  EFIBEAlertError = class( Exception);

implementation

resourcestring
  SNoEventsRegistered  = 'You must register events before queueing them';
  SInvalidCancellation = 'You cannot call CancelEvents from within an OnEventAlert handler';
  SInvalidEvent        = 'Invalid blank event added to EventAlerter events list';
  SInvalidQueueing     = 'You cannot call QueueEvents from within an OnEventAlert handler';
  SInvalidRegistration = 'You cannot Register or Unregister events from within an OnEventAlert handler';  SMaximumEvents       = 'You can only register 15 events per EventAlerter';

// TFIBEventAlerter

procedure HandleEvent( param: integer); stdcall;
begin
  // don't let exceptions propogate out of thread
  try
    TFIBEventAlerter( param).HandleEvent;
  except
    Application.HandleException( nil);
  end;
end;

procedure IBEventCallback( ptr: pointer; length: short; updated: PChar); cdecl;
var
  ThreadID: DWORD;
begin
  // Handle events asynchronously in second thread
  EnterCriticalSection( TFIBEventAlerter( ptr).CS);
  TFIBEventAlerter( ptr).UpdateResultBuffer( length, updated);
  if TFIBEventAlerter( ptr).Queued then
    CloseHandle( CreateThread( nil, 8192, @HandleEvent, ptr, 0, ThreadID));
  LeaveCriticalSection( TFIBEventAlerter( ptr).CS);
end;

constructor TFIBEventAlerter.Create( AOwner: TComponent);
begin
  inherited Create( AOwner);
  InitializeCriticalSection( CS);
  FEvents := TStringList.Create;
  with TStringList( FEvents) do
  begin
    OnChange := EventChange;
    Duplicates := dupIgnore;
  end;
end;

destructor TFIBEventAlerter.Destroy;
begin
  UnregisterEvents;
  SetDatabase( nil);
  TStringList(FEvents).OnChange := nil;
  FEvents.Free;
  DeleteCriticalSection( CS);
  inherited Destroy;
end;

procedure TFIBEventAlerter.ValidateDatabase(Database: TFIBDatabase);
begin
  if not Assigned(Database) or not Database.Connected then
    FIBError(feDatabaseClosed, [nil]);
end;

procedure TFIBEventAlerter.CancelEvents;
var
  dbHandle: TISC_DB_HANDLE;
begin
  if ProcessingEvents then
    raise EFIBEAlertError.Create( SInvalidCancellation);
  if FQueued then
  begin
    try
      // wait for event handler to finish before cancelling events
      EnterCriticalSection( CS);
      ValidateDatabase( Database);
      FQueued := false;
      Changing := true;
      dbHandle := Database.Handle;
      if Isc_Cancel_Events( StatusVector, @dbHandle, @EventID) <> 0 then
        IBError;
    finally
      LeaveCriticalSection( CS);
    end;
  end;
end;

procedure TFIBEventAlerter.DoQueueEvents;
var
  callback: pointer;
  dbHandle: Tisc_db_handle;
begin
  ValidateDatabase( DataBase);
  callback := @IBEventCallback;
  dbHandle := Database.Handle;
  if Isc_Que_Events( StatusVector, @dbHandle, @EventID, EventBufferLen,
        EventBuffer, Tisc_callback(callback), PVoid(self)) <> 0 then
    IBError;
  FQueued := true;
end;

procedure TFIBEventAlerter.EventChange( sender: TObject);
begin
  // check for blank event
  if TStringList(Events).IndexOf( '') <> -1 then
    raise EFIBEAlertError.Create( SInvalidEvent);
  // check for too many events
  if Events.Count > MaxEvents then
  begin
    TStringList(Events).OnChange := nil;
    Events.Delete( MaxEvents);
    TStringList(Events).OnChange := EventChange;
    raise EFIBEAlertError.Create( SMaximumEvents);
  end;
  if Registered then RegisterEvents;
end;

procedure TFIBEventAlerter.HandleEvent;
var
  CancelAlerts: Boolean;
  i: integer;
  status: PISC_STATUS;
begin
  try
    // prevent modification of vital data structures while handling events
    EnterCriticalSection( CS);
    ProcessingEvents := true;
    status:=StatusVector;
    Isc_Event_Counts( status, EventBufferLen, EventBuffer, ResultBuffer);
    CancelAlerts := false;
    if assigned(FOnEventAlert) and not Changing then
    begin
      for i := 0 to Events.Count-1 do
      begin
        try
          if (status^ <> 0) and not CancelAlerts then
            FOnEventAlert( self, Events[Events.Count-i-1], status^, CancelAlerts);
        except
          Application.HandleException( nil);
        end;
        Inc(status);
      end;
    end;
    Changing := false;
    if not CancelAlerts and FQueued then DoQueueEvents;
  finally
    ProcessingEvents := false;
    LeaveCriticalSection( CS);
  end;
end;

procedure TFIBEventAlerter.Loaded;
begin
  inherited Loaded;
  try
    if RegisteredState then RegisterEvents;
  except
    if csDesigning in ComponentState then
      Application.HandleException( self)
    else raise;
  end;
end;

procedure TFIBEventAlerter.Notification( AComponent: TComponent;
                                        Operation: TOperation);
begin
  inherited Notification( AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FDatabase) then
  begin
    UnregisterEvents;
    FDatabase := nil;
  end;
end;

procedure TFIBEventAlerter.QueueEvents;
begin
  if not FRegistered then
    raise EFIBEAlertError.Create( SNoEventsRegistered);
  if ProcessingEvents then
    raise EFIBEAlertError.Create( SInvalidQueueing);
  if not FQueued then
  begin
    try
      // wait until current event handler is finished before queuing events
      EnterCriticalSection( CS);
      DoQueueEvents;
      Changing := true;
    finally
      LeaveCriticalSection( CS);
    end;
  end;
end;

procedure TFIBEventAlerter.RegisterEvents;
var
  i: integer;
  bufptr: pointer;
  eventbufptr: pointer;
  resultbufptr: pointer;
  buflen: integer;
begin
  ValidateDatabase( Database);
  if csDesigning in ComponentState then FRegistered := true
  else begin
    UnregisterEvents;
    if Events.Count = 0 then exit;
    for i := 0 to Events.Count-1 do
      StrPCopy( @Buffer[i][0], Events[i]);
    i := Events.Count;
    bufptr := @buffer[0];
    eventbufptr :=  @EventBuffer;
    resultBufPtr := @ResultBuffer;
    asm
      mov ecx, dword ptr [i]
      mov eax, dword ptr [bufptr]
      @@1:
      push eax
      add  eax, EventLength
      loop @@1
      push dword ptr [i]
      push dword ptr [resultBufPtr]
      push dword ptr [eventBufPtr]
      call [Isc_Event_Block]
      mov  dword ptr [bufLen], eax
      mov eax, dword ptr [i]
      shl eax, 2
      add eax, 12
      add esp, eax
    end;
    EventBufferlen := Buflen;
    FRegistered := true;
    QueueEvents;
  end;
end;

procedure TFIBEventAlerter.SetEvents( value: TStrings);
begin
  FEvents.Assign( value);
end;

procedure TFIBEventAlerter.SetDatabase( value: TFIBDatabase);
begin
  if value <> FDatabase then
  begin
    UnregisterEvents;
    if assigned( value) and value.Connected then ValidateDatabase( value);
    FDatabase := value;
  end;
end;

procedure TFIBEventAlerter.SetRegistered( value: Boolean);
begin
  if (csReading in ComponentState) then
    RegisteredState := value
  else if FRegistered <> value then
    if value then RegisterEvents else UnregisterEvents;
end;

procedure TFIBEventAlerter.UnregisterEvents;
begin
  if ProcessingEvents then
    raise EFIBEAlertError.Create( SInvalidRegistration);
  if csDesigning in ComponentState then
    FRegistered := false
  else if not (csLoading in ComponentState) then
  begin
    CancelEvents;
    if FRegistered then
    begin
      Isc_Free( EventBuffer);
      EventBuffer := nil;
      Isc_Free( ResultBuffer);
      ResultBuffer := nil;
    end;
    FRegistered := false;
  end;
end;

procedure TFIBEventAlerter.UpdateResultBuffer( length: short; updated: PChar);
var
  i: integer;
begin
  for i := 0 to length-1 do
    ResultBuffer[i] := updated[i];
end;

end.
