unit BlobFncs;

interface

uses
  SysUtils, ibase, ib_externals, udf_glob, StdFuncs;

// Information functions
function BlobMaxSegmentLength(Blob: PBlob): Integer; cdecl; export;
function BlobSegmentCount(Blob: PBlob): Integer; cdecl; export;
function BlobSize(Blob: PBlob): Integer; cdecl; export;
// String-ish functions
{* Obviously, these functions should be used with care, and only when you are
   confident that your blob is not overly large.
   Since varchars can go all the way to 32767 (I think), I find
   Blobs to be somewhat useless as containers for large strings, but
   here are some functions to do it anyways... }
function BlobAsPChar(Blob: PBlob): PChar; cdecl; export;
function BlobLeft(Blob: PBlob; var Number: Integer): PChar; cdecl; export;
function BlobLine(Blob: PBlob; var Number: Integer): PChar; cdecl; export;
function BlobMid(Blob: PBlob; var Start, Number: Integer): PChar; cdecl; export;
function BlobRight(Blob: PBlob; var Number: Integer): PChar; cdecl; export;
function StrBlob(sz: PChar; Blob: PBlob): PBlob; cdecl; export;
function BlobBinCmp(BlobA: PBlob; BlobB: Pblob): Integer; cdecl; export;

implementation

const
  CRLF = #13 + #10;

function BlobMaxSegmentLength(Blob: PBlob): Integer; cdecl; export;
begin
  {$ifdef FULDebug}
  WriteDebug('BlobMaxSegmentLength() - Enter');
  {$endif}
  result := 0;
  if (not Assigned(Blob)) or
     (not Assigned(Blob^.BlobHandle)) then exit;
  result := Blob^.MaxSegmentLength;
  {$ifdef FULDebug}
  WriteDebug('BlobMaxSegmentLength() - Exit');
  {$endif}
end;

function BlobSegmentCount(Blob: PBlob): Integer; cdecl; export;
begin
  {$ifdef FULDebug}
  WriteDebug('BlobSegmentCount() - Enter');
  {$endif}
  result := 0;
  if (not Assigned(Blob)) or
     (not Assigned(Blob^.BlobHandle)) then exit;
  result := Blob^.SegmentCount;
  {$ifdef FULDebug}
  WriteDebug('BlobSegmentCount() - Exit');
  {$endif}
end;

function BlobSize(Blob: PBlob): Integer; cdecl; export;
begin
  {$ifdef FULDebug}
  WriteDebug('BlobSize() - Enter');
  {$endif}
  result := 0;
  if (not Assigned(Blob)) or
     (not Assigned(Blob^.BlobHandle)) then exit;
  result := Blob^.TotalSize;
  {$ifdef FULDebug}
  WriteDebug('BlobSize() - Exit');
  {$endif}
end;

function BlobAsPChar(Blob: PBlob): PChar; cdecl; export;
var
  bytes_read, bufsize: ushort;
  total_bytes_read: Long;
  res: short;
  st: String;
begin
  {$ifdef FULDebug}
  WriteDebug('BlobAsPChar() - Enter');
  {$endif}
  result := nil;
  if (not Assigned(Blob)) or
     (not Assigned(Blob^.BlobHandle)) then exit;
  with Blob^ do begin
    // bytes_left := TotalSize;                // I have TotalSize bytes to read.
    bufsize:=MaxSegmentLength; // char and varchar can't handle more bytes;
    if (TotalSize = 0) then exit;          // if I've nothing to copy, exit.
    SetString(st, nil, TotalSize);         // read whole blob
    total_bytes_read := 0;                  // total bytes read is 0.
    repeat
      // Using BlobHandle, store at most "bytes_left" bytes in
      //   the buffer starting where I last left off
      res:=GetSegment(BlobHandle, @st[total_bytes_read + 1], bufsize, bytes_read);
      // Increment total_bytes_read by the number of bytes actually read.
      Inc(total_bytes_read, bytes_read);
    until TotalSize <= total_bytes_read;
  end;
  result := MakeResultString(PChar(st), nil, 0);
  {$ifdef FULDebug}
  WriteDebug('BlobAsPChar() - Exit');
  {$endif}
end;

function BlobLeft(Blob: PBlob; var Number: Integer): PChar;
var
  bytes_read, bufsize: ushort;
  total_bytes_read: Long;
  st: String;
begin
  {$ifdef FULDebug}
  WriteDebug('BlobLeft() - Enter');
  {$endif}
  result := nil;
  if (not Assigned(Blob)) or
     (not Assigned(Blob^.BlobHandle)) or
     (Number < 0) then exit;
  with Blob^ do begin
    total_bytes_read := 0;                  // total bytes read is 0.
    bufsize := Min(MaxSegmentLength, Number);
    if (TotalSize = 0) then exit;          // if I've nothing to copy, exit.
    SetString(st, nil, Number);            // as many as we want to read
    //st[bytes_left] := #0;               // Set the terminating null value.
    repeat
      // Using BlobHandle, store at most "bytes_left" bytes in
      //   the buffer starting where I last left off
      GetSegment(BlobHandle, @st[total_bytes_read + 1], bufsize, bytes_read);
      // Increment total_bytes_read by the number of bytes actually read.
      Inc(total_bytes_read, bytes_read);
    until bufsize <= total_bytes_read;
  end;
  result := MakeResultString(PChar(st), nil, 0);
  {$ifdef FULDebug}
  WriteDebug('BlobLeft() - Exit');
  {$endif}
end;

// Lines are zero based.
function BlobLine(Blob: PBlob; var Number: Integer): PChar;
var
  i, j: Integer;
begin
  { First read the whole blob into a character buffer, then
    scan through the entire blob, reusing the first part of the
    buffer to write the Number-th line of the Blob. }
  {$ifdef FULDebug}
  WriteDebug('BlobLine() - Enter');
  {$endif}
  result := BlobAsPChar(Blob);
  if (result = nil) or (result[0] = #0) then exit;
  j := -1;
  while (Number >= 0) and ((j = -1) or (result[j] <> #0)) do begin
    i := -1;
    repeat
      Inc(i); Inc(j);
      // Here only write to the return string if j > i.
      if (j > i) then
        result[i] := result[j];
    until (result[j] in [#0, #10, #13]);
    // if current character is cr then check to see if there
    // is a line feed following it.
    if (result[j] in [#10, #13]) then begin
      result[i] := #0; // "End" the current line.
      if (result[j+1] = #10) then Inc(j);
    end;
    Dec(Number);
  end;
  if Number <> -1 then result[0] := #0;
  {$ifdef FULDebug}
  WriteDebug('BlobLine() - Exit');
  {$endif}
end;

function BlobMid(Blob: PBlob; var Start, Number: Integer): PChar;
begin
  {$ifdef FULDebug}
  WriteDebug('BlobMid() - Enter');
  {$endif}
  result := nil;
  if (Start < 0) or (Number < 0) then exit;
  //result := BlobAsPChar(Blob);
  result := BlobLeft(Blob, Number); // avoid reading more
  result := MakeResultString(
    PChar(Copy(result, Start + 1, Number)), result, 0);
  {$ifdef FULDebug}
  WriteDebug('BlobMid() - Exit');
  {$endif}
end;

function BlobRight(Blob: PBlob; var Number: Integer): PChar;
begin
  {$ifdef FULDebug}
  WriteDebug('BlobRight() - Enter');
  {$endif}
  result := nil;
  if Number < 0 then exit;
  result := BlobAsPChar(Blob);
  result := MakeResultString(
    PChar(Copy(result, Length(result) - Number + 1, Number)), result, 0);
  {$ifdef FULDebug}
  WriteDebug('BlobRight() - Exit');
  {$endif}
end;

// According to IB docs, this is how you do it for blobs.
// First pass the "first" argument
// Second, pass the "return" argument (A Blob) as the last argument.
// then you would just declare the external function in
// IB as shown below
function StrBlob(sz: PChar; Blob: PBlob): PBlob;
begin
  {$ifdef FULDebug}
  WriteDebug('StrBlob() - Enter');
  {$endif}
  result := Blob;
  if (not Assigned(Blob)) or
     (not Assigned(Blob^.BlobHandle)) then exit;
  Blob^.PutSegment(Blob^.BlobHandle, sz, StrLen(sz));
  {$ifdef FULDebug}
  WriteDebug('StrBlob() - Exit');
  {$endif}
end;

function BlobBinCmp(BlobA: PBlob; BlobB: Pblob): Integer; cdecl; export;
var
//  bytes_readA, bytes_leftA, total_bytes_readA, segmA: Long;
//  bytes_readB, bytes_leftB, total_bytes_readB, segmB: Long;
  bytes_readA, bufsizeA: ushort; total_bytes_readA, segmA: Long;
  bytes_readB, bufsizeB: ushort; total_bytes_readB, segmB: Long;
  iterator: Long;
  stA, stB: String;
begin
  {$ifdef FULDebug}
  WriteDebug('BlobBinCmp() - Enter');
  {$endif}
  result := -1;
  if (not Assigned(BlobA)) or
     (not Assigned(BlobA^.BlobHandle)) then exit;
  if (not Assigned(BlobB)) or
     (not Assigned(BlobB^.BlobHandle)) then exit;
  if (BlobA.TotalSize <> BlobB.TotalSize)
  then begin
     result := min(BlobA.TotalSize, BlobB.TotalSize) + 1;
     {$ifdef FULDebug}
     WriteDebugV(['ASize-BSize', BlobA.TotalSize, '-', BlobB.TotalSize]);
     WriteDebug('BlobBinCmp() - Exit');
     {$endif}
     Exit;
  end;
  result := 0;
  if (BlobA.TotalSize = 0) and (BlobB.TotalSize = 0)
  then begin
    {$ifdef FULDebug}
    WriteDebug('BlobBinCmp() - Exit');
    {$endif}
    Exit;
  end;
  with BlobA^ do begin
    bufsizeA := MaxSegmentLength;                // I have TotalSize bytes to read.
    SetString(stA, nil, MaxSegmentLength);
    total_bytes_readA := 0;
    segmA := MaxSegmentLength;
    {$ifdef FULDebug}
    WriteDebugV(['ASizeMaxSegmentLength', TotalSize, '-', MaxSegmentLength]);
    {$endif}
  end;
  with BlobB^ do begin
    bufsizeB := MaxSegmentLength;                // I have TotalSize bytes to read.
    SetString(stB, nil, MaxSegmentLength);
    total_bytes_readB := 0;
    segmB := MaxSegmentLength;
    {$ifdef FULDebug}
    WriteDebugV(['BSizeMaxSegmentLength', TotalSize, '-', MaxSegmentLength]);
    {$endif}
  end;
  iterator := 0;
  repeat
    // Using BlobHandle, store at most "bytes_left" bytes in
    //   the buffer starting where I last left off
    if iterator = total_bytes_readA
    then begin
      {$ifdef FULDebug}
      WriteDebugV(['iterator=totalReadA', total_bytes_readA]);
      {$endif}
      With BlobA^ do
        GetSegment(BlobHandle, @stA[1], bufsizeA, bytes_readA);
      // Increment total_bytes_read by the number of bytes actually read.
      Inc(total_bytes_readA, bytes_readA);
    end;
    if iterator = total_bytes_readB
    then begin
      {$ifdef FULDebug}
      WriteDebugV(['iterator=totalReadB', total_bytes_readB]);
      {$endif}
      With BlobB^ do
        GetSegment(BlobHandle, @stB[1], bufsizeB, bytes_readB);
      // Increment total_bytes_read by the number of bytes actually read.
      Inc(total_bytes_readB, bytes_readB);
    end;
    {$ifdef FULDebug}
    WriteDebugV(['[itA,itB]=',
      stA[iterator mod segmA + 1], stB[iterator mod segmB + 1]]);
    {$endif}
    if stA[iterator mod segmA + 1] <> stB[iterator mod segmB + 1]
    then begin
    	result := iterator + 1;
      break;
    end;
    Inc(iterator);
    {$ifdef FULDebug}
    WriteDebugV(['Iterator=', iterator]);
    {$endif}
  until (iterator = BlobA.TotalSize) or (iterator = BlobB.TotalSize);
  {$ifdef FULDebug}
  WriteDebug('BlobBinCmp() - Exit');
  {$endif}
end;

end.

