{************************************************************************}
{                                                                        }
{   "gb_DataSets Components" - set of  additional components for IBX and }
{                             FIBPLUS libraries.                         }
{                                                                        }
{     These components are written to make possible a normal             }
{     navigation on potentially big tables or queries returning          }
{     a potentially high count of records                                }
{                                                                        }
{    ATTENTION: Don't forget about the rule of DOUBLE INDEXES for normal }
{               perfomance (see readme.txt for more information )!!!     }
{                                                                        }
{   Copyright (c) 2001, 2004 Spirin Sergey                               }
{                                                                        }
{   Company        : Paritet Soft LTD, Moscow                            }
{   Support e-mail : spirin@paritetsoft.ru                               }
{                                                                        }
{   Please see the file license.txt for full license information         }
{                                                                        }
{************************************************************************}

unit gb_CustomDataSet;

{$I gb_DataInc.inc}

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, DB,
{$IFDEF VER_IBX}
  IBExternals, IBHeader,  IBSQL, IBDataBase, IBIntf, IBBlob;
{$ELSE}
  fibdatabase, fibquery, ibase, ib_intf, ib_externals, FIBMiscellaneous;
{$ENDIF}

type
  TLocateOptionExt = (leCaseInsensitive, lePartialKey, leContaining);
  TLocateOptionsExt = set of TLocateOptionExt;

  TInternalLocateOption = (ioNearest, ioNextFlag, ioPriorFlag);
  TInternalLocateOptions = set of TInternalLocateOption;

  TNavigateOption = (opUseFirstStatement, opUseCyrCollate, opCheckNullsInRefreshWhere);
  TNavigateOptions = set of TNavigateOption;

  TUpdateOption = (opCommitOnPost, opCheckNullsInWhere, opLockRecOnEdit, opWhereChangeOnInsert);
  TUpdateOptions = set of TUpdateOption;

  TGeneratorType = (gtOnInsert, gtOnEdit, gtOnInsertEdit, gtServer, gtOnNewRecord);
  TGeneratorTypes = set of TGeneratorType;

  TRecInfo = record
    Bookmark: LongInt;
    BookmarkFlag: TBookmarkFlag;
  end;
  PRecInfo = ^TRecInfo;

  TFieldInfo = record
    FOffset, FDataSize, FBookmarkOffset: LongInt;
    FType: Short;
    FScale, FHoleNumber: Short;
    FUnknownType, FNotUpdateble: Boolean;
    FFieldName: string[67];
    FRelationNameForOrder: string[135];
  end;
  PFieldInfo = ^TFieldInfo;

  TGeneratorDef = class(TCollectionItem)
  private
    FFieldName: string;
    FGeneratorName: string;
    FGeneratorType: TGeneratorType;
    FIncrementBy: integer;
  protected
    function GetDisplayName: string; override;
  public
    constructor Create(Collection: TCollection); override;
  published
    property FieldName: string read FFieldName write FFieldName;
    property GeneratorName: string read FGeneratorName write FGeneratorName;
    property GeneratorType: TGeneratorType read FGeneratorType write FGeneratorType default gtOnInsert;
    property IncrementBy: integer read FIncrementBy write FIncrementBy default 1;
  end;

  TGeneratorDefs = class(TOwnedCollection)
  private
    function GetGeneratorDef(Index: Integer): TGeneratorDef;
    procedure SetGeneratorDef(Index: Integer; Value: TGeneratorDef);
  public
    property Items[Index: Integer]: TGeneratorDef read GetGeneratorDef write SetGeneratorDef; default;
  end;

  TgbMacro = class(TCollectionItem)
  private
    FMacroName, FMacroValue: string;
  protected
    function GetDisplayName: string; override;
  published
    property MacroName: string read FMacroName write FMacroName;
    property MacroValue: string read FMacroValue write FMacroValue;
  end;

  TgbMacros = class(TOwnedCollection)
  private
    function GetMacro(Index: Integer): TgbMacro;
    procedure SetMacro(Index: Integer; Value: TgbMacro);
  public
    property Items[Index: Integer]: TgbMacro read GetMacro write SetMacro; default;
  end;

  TgbCustomDataSet = class(TDataSet)
  private
    FBase: {$IFDEF VER_IBX} TIBBase; {$ELSE} TFIBBase; {$ENDIF}
    CurrentSelect, FQSelect, FQSelectPart, FQSelectDescPart, FQSelectDesc,
    FTecknics, FQRefresh, FQLocate: {$IFDEF VER_IBX} TIBSQL; {$ELSE} TFIBQuery; {$ENDIF}
    FPostTransaction: {$IFDEF VER_IBX} TIBTransaction;{$ELSE} TFIBTransaction; {$ENDIF}
    FOpen, FFirstIsReal, FLastIsReal, FStreamedActive, FRefreshWithNextNeed, FReadOnly,
    FUseMacros: Boolean;
    FBufferChunks: SmallInt;
    FRecordSize, FRecordBufferSize, FRecordCount, FCurrentRecord, FAboveDesc,
    FCalcFieldsOffset: Integer;
    FInsertText, FDeleteText, FUpdateText, FRefreshText, FKeyFields: TStringList;
    TempBuffer: PChar;
    FCache: array of PChar;
    FSQLWaitCursor: TCursor;
    MapOfRecord, FWriteBlobs: TList;
    FBeforeDatabaseDisconnect, FAfterDatabaseDisconnect, FDatabaseFree,
    FBeforeTransactionEnd, FAfterTransactionEnd, FTransactionFree: TNotifyEvent;
    FNavOptions: TNavigateOptions;
    FUpdateOptions: TUpdateOptions;
    FGeneratorDefs: TGeneratorDefs;
    FSQLParams: {$IFDEF VER_IBX} TIBXSQLDA {$ELSE} TFIBXSQLDA {$ENDIF};
{$IFDEF VER_IBX}
    FGDSLibrary: IGDSLibrary;
{$ELSE}
    FGDSLibrary: IIBClientLibrary;
{$ENDIF}
    procedure AddToMapOfRecord;
    function CheckBmkRelative(Bmk: TBookmark; RecNum: integer): Boolean;
    procedure CheckCacheState(AFree: Boolean);
    function CheckForNulls(SQL: TStrings; CheckType: TUpdateOption; OldBufferNeed: Boolean): string;
    procedure CheckGenerators(GenType: TGeneratorTypes);
    function CheckRelation(const S: string; ACheck: Boolean): string;
    procedure ClearMapOfRecord;
    procedure ClearWriteBlobs;
    procedure CorrectTop(Above: integer);
    function DescNow: Boolean;
    procedure DoAfterDatabaseDisconnect(Sender: TObject);
    procedure DoAfterTransactionEnd(Sender: TObject);
    procedure DoBeforeDatabaseDisconnect(Sender: TObject);
    procedure DoBeforeTransactionEnd(Sender: TObject);
    procedure DoDatabaseFree(Sender: TObject);
    procedure DoTransactionFree(Sender: TObject);
    procedure ExecWithCursor(IBSQL: {$IFDEF VER_IBX} TIBSQL); {$ELSE} TFIBQuery); {$ENDIF}
    procedure FinalizeWriteBlobs;
    function GenerateWhere(SList: TStrings; NoDesc: Boolean): string;
    function GetActiveBuf: PChar;
    function GetDatabase: {$IFDEF VER_IBX} TIBDatabase; {$ELSE} TFIBDatabase; {$ENDIF}
    function GetDeleteSQL: TStrings;
    procedure GetFieldsInOrder_(SQL, SList: TStrings);
    function GetInsertSQL: TStrings;
    function GetMapByName(const FieldName: string): PFieldInfo;
    function GetPosOfStatement(const SQLText, Statement: string; var NextPos: integer): integer;
    function GetRefreshSQL: TStrings;
    function GetSelectSQL: TStrings;
    function GetTransaction: {$IFDEF VER_IBX} TIBTransaction; {$ELSE} TFIBTransaction; {$ENDIF}
    function GetUpdateSQL: TStrings;
    function InternalLocateExt(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptionsExt; IntOptions: TInternalLocateOptions): Boolean;
    function IsStatement(const S: string): Boolean;
    function LockCurrentRecord: Boolean;
    procedure MoveInCache(CurIndex, NewIndex: Integer);
    procedure OrderToDesc(SQL, SList: TStrings);
    procedure ReadFromCache(ARecNo: integer; Buffer: PChar);
    function RefreshAround(IBSQL: {$IFDEF VER_IBX} TIBSQL; {$ELSE} TFIBQuery; {$ENDIF} Center: Boolean): Boolean;
    procedure RefreshCurrentRecord(InPostTransaction: Boolean);
    procedure ReorderCache(Direction: Boolean);
    procedure SetDatabase(Value: {$IFDEF VER_IBX} TIBDatabase); {$ELSE} TFIBDatabase); {$ENDIF}
    procedure SetDeleteSQL(Value: TStrings);
    procedure SetInsertSQL(Value: TStrings);
    procedure SetLocateWhere(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptionsExt; IntOptions: TInternalLocateOptions; IBSQL: {$IFDEF VER_IBX} TIBSQL); {$ELSE} TFIBQuery); {$ENDIF}
    procedure SetNextPriorWhere(SQL: TStrings; ANext: Boolean);
    procedure SetParamsTo(IBSQL, From: {$IFDEF VER_IBX} TIBSQL; {$ELSE} TFIBQuery; {$ENDIF} FromBuffer: PChar; OldBufferNeed: Boolean);
    procedure SetSelectSQL(Value: TStrings);
    procedure SetTransaction(const Value: {$IFDEF VER_IBX} TIBTransaction); {$ELSE} TFIBTransaction); {$ENDIF}
    procedure SetUpdateSQL(Value: TStrings);
    function SynchronizeSQL(NumInArray: integer): TStrings; virtual; abstract;
    function TokenNext(const S: string; BlankInt: Boolean; var CurPos, CurEnd: integer):string;
    procedure WriteToCache(ARecNo: integer; Buffer: PChar);
  protected
    FLockSQLText, FKeyFieldsForBookmark : string;
    procedure ActivateConnection;
    procedure ActivateTransaction;
    function AllocRecordBuffer: PChar; override;
    function CheckNext(Buffer: PChar): TGetResult;
    function CheckPrior(Buffer: PChar): TGetResult;
    procedure ClearCalcFields(Buffer: PChar); override;
{$IFDEF VER_FIBPLUS}
    procedure CloseSelectQuery;
{$ENDIF}
    procedure ConvertSQLs(ReAssign: Boolean); virtual;
    procedure DoAfterPost; override;
    procedure DoBeforePost; override;
    procedure DoOnNewRecord; override;
    procedure FetchRecordToBuffer(Qry: {$IFDEF VER_IBX} TIBSQL; {$ELSE} TFIBQuery; {$ENDIF} ARecNo: integer; Buffer: PChar);
    procedure FreeRecordBuffer(var Buffer: PChar); override;
    procedure GetBookmarkData(Buffer: PChar; Data: Pointer); override;
    function GetBookmarkFlag(Buffer: PChar): TBookmarkFlag; override;
    function GetCanModify: Boolean; override;
    function GetMap(I: integer; ACheck: Boolean): PFieldInfo;
    function GetMapCount: integer;
    function GetMasterFieldValue(const ParName: string): Variant; virtual; abstract;
    function GetRecord(Buffer: PChar; GetMode: TGetMode; DoCheck: Boolean): TGetResult; override;
    function GetRecordSize: Word; override;
    procedure InsertWhere(const S: string; SQL: TStrings);
    procedure InternalCancel; override;
    procedure InternalClose; override;
    procedure InternalDelete; override;
    procedure InternalEdit; override;
    procedure InternalFirst; override;
    procedure InternalGotoBookmark(Bookmark: Pointer); override;
    procedure InternalHandleException; override;
    procedure InternalInitFieldDefs; override;
    procedure InternalInitRecord(Buffer: PChar); override;
    procedure InternalInsert; override;
    procedure InternalLast; override;
    procedure InternalOpen; override;
    procedure InternalPost; override;
    procedure InternalRefresh; override;
    procedure InternalSetToRecord(Buffer: PChar); override;
    function IsCursorOpen: Boolean; override;
    procedure Loaded; override;
    procedure SetActive(Value: Boolean); override;
    procedure SetBookmarkData(Buffer: PChar; Data: Pointer); override;
    procedure SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag); override;
    procedure SetFieldData(Field : TField; Buffer : Pointer); override;
    procedure SetFieldData(Field : TField; Buffer : Pointer; NativeFormat : Boolean); overload; override;
    procedure SetFirstParams; virtual;
    procedure SetRefreshSQL(Value: TStrings);
    property DeleteSQL: TStrings read GetDeleteSQL write SetDeleteSQL;
    property InsertSQL: TStrings read GetInsertSQL write SetInsertSQL;
    property RefreshSQL: TStrings read GetRefreshSQL write SetRefreshSQL;
    property SelectSQL: TStrings read GetSelectSQL write SetSelectSQL;
    property UpdateSQL: TStrings read GetUpdateSQL write SetUpdateSQL;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function CompareBookmarks(Bookmark1, Bookmark2: TBookmark): Integer; override;
    function CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream; override;
    function GetFieldData(Field : TField; Buffer : Pointer) : Boolean; overload; override;
    function GetFieldData(Field: TField; Buffer: Pointer; NativeFormat: Boolean): Boolean; overload; override;
    procedure GetFieldsInOrder(Fields: TStrings);
    function GetQSelect: {$IFDEF VER_IBX} TIBSQL; {$ELSE} TFIBQuery; {$ENDIF}
    function IsSequenced: Boolean; override;
    function Locate(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptions): Boolean; override;
    function LocateExt(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptionsExt): Boolean;
    function LocateNearest(const KeyFields: string; const KeyValues: Variant): Boolean;
    function LocateNext(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptionsExt): Boolean;
    function LocatePrior(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptionsExt): Boolean;
    function Lookup(const KeyFields: string; const KeyValues: Variant;
      const ResultFields: string): Variant; override;
    procedure ReQuery(KeepPosition: Boolean);
    property PostTransaction: {$IFDEF VER_IBX} TIBTransaction {$ELSE} TFIBTransaction {$ENDIF} read FPostTransaction;
  published
    property AfterDatabaseDisconnect: TNotifyEvent read FAfterDatabaseDisconnect write FAfterDatabaseDisconnect;
    property AfterTransactionEnd: TNotifyEvent read FAfterTransactionEnd write FAfterTransactionEnd;
    property BeforeDatabaseDisconnect: TNotifyEvent read FBeforeDatabaseDisconnect write FBeforeDatabaseDisconnect;
    property BeforeTransactionEnd: TNotifyEvent read FBeforeTransactionEnd write FBeforeTransactionEnd;
    property BufferChunks: SmallInt read FBufferChunks write FBufferChunks Default 100;
    property Database: {$IFDEF VER_IBX} TIBDatabase {$ELSE} TFIBDatabase {$ENDIF} read GetDatabase write SetDatabase;
    property DatabaseFree: TNotifyEvent read FDatabaseFree write FDatabaseFree;
    property GeneratorDefs: TGeneratorDefs read FGeneratorDefs write FGeneratorDefs;
    property Options: TNavigateOptions read FNavOptions write FNavOptions default [opCheckNullsInRefreshWhere];
    property ReadOnly: Boolean read FReadOnly write FReadOnly default False;
    property SQLWaitCursor: TCursor read FSQLWaitCursor write FSQLWaitCursor Default crSQLWait;
    property Transaction: {$IFDEF VER_IBX} TIBTransaction {$ELSE} TFIBTransaction {$ENDIF} read GetTransaction write SetTransaction;
    property TransactionFree: TNotifyEvent read FTransactionFree write FTransactionFree;
    property UpdateOptions: TUpdateOptions read FUpdateOptions write FUpdateOptions default [opCommitOnPost, opCheckNullsInWhere, opLockRecOnEdit, opWhereChangeOnInsert];
    property Active;
    property AutoCalcFields;
    property BeforeOpen;
    property AfterOpen;
    property BeforeClose;
    property AfterClose;
    property BeforeInsert;
    property AfterInsert;
    property BeforeEdit;
    property AfterEdit;
    property BeforePost;
    property AfterPost;
    property BeforeCancel;
    property AfterCancel;
    property BeforeDelete;
    property AfterDelete;
    property BeforeScroll;
    property AfterScroll;
    property BeforeRefresh;
    property AfterRefresh;
    property OnCalcFields;
    property OnDeleteError;
    property OnEditError;
    property OnNewRecord;
    property OnPostError;
  end;

  TgbDataSet = class;

  TMacrosDef = class(TPersistent)
  private
    FOwner: TgbDataSet;
    FMacros: TgbMacros;
    function GetUseMacros: Boolean;
    procedure SetUseMacros(Value: Boolean);
    function GetMacroChar: Char;
    procedure SetMacroChar(const Value: Char);
  public
    constructor Create(AOwner: TComponent);
    function  GetOwner: TPersistent; override;
  published
    property UseMacros: Boolean read GetUseMacros write SetUseMacros default False;
    property MacroChar: Char read GetMacroChar write SetMacroChar default '%';
    property Macros: TgbMacros read FMacros write FMacros;
  end;

  TgbDataSet = class(TgbCustomDataSet)
  private
    FParamsAssigned, FQSelectAlreadyInArray, FSQLSaved: Boolean;
    FMacroChar: Char;
    FPlanForDescQuerys: string;
    FMacrosDef: TMacrosDef;
    FSavedSQLs: array[0..4] of string;
    FSynchrList: TStringList;
    FCurrentSQLInSynchr: integer;
    procedure ActivateFQSelect;
    procedure AssignParams;
    function GetSQLParams: {$IFDEF VER_IBX} TIBXSQLDA; {$ELSE} TFIBXSQLDA; {$ENDIF}
    function SynchronizeSQL(NumInArray: integer): TStrings; override;
    procedure SQLChanging(Sender: TObject);
    procedure SynchrChanged(Sender: TObject);
  protected
    procedure ConvertSQLs(ReAssign: Boolean); override;
    procedure ExpandMacros;
    function ExpandSQL(const SQL: string): string;
    procedure InternalClose; override;
    procedure InternalOpen; override;
    procedure SetFirstParams; override;
    procedure InternalInitFieldDefs; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function MacroByName(Idx: string): TgbMacro;
    function ParamByName(Idx: string): {$IFDEF VER_IBX} TIBXSQLVAR; {$ELSE} TFIBXSQLVAR; {$ENDIF}
    property Params: {$IFDEF VER_IBX} TIBXSQLDA {$ELSE} TFIBXSQLDA {$ENDIF} read GetSQLParams;
  published
    property DeleteSQL;
    property InsertSQL;
    property KeyFieldsForBookmark: string read FKeyFieldsForBookmark write FKeyFieldsForBookmark;
    property MacrosDef: TMacrosDef read FMacrosDef write FMacrosDef;
    property PlanForDescQuerys: string read FPlanForDescQuerys write FPlanForDescQuerys;
    property RefreshSQL;
    property SelectSQL;
    property UpdateSQL;
  end;

  TgbIBBlobStream = class{$IFDEF VER_IBX}(TIBBlobStream){$ELSE}(TFIBBlobStream){$ENDIF}
  private
    FField: TField;
  end;

  TgbBlobStream = class(TStream)
  protected
    FField: TField;
    FBlobStream: TgbIBBlobStream;
    FModified : Boolean;
  public
    constructor Create(AField: TField; ABlobStream: TgbIBBlobStream; Mode: TBlobStreamMode);
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    procedure SetSize(NewSize: Longint); override;
    function Write(const Buffer; Count: Longint): Longint; override;
  end;

function ExtractIdent(Ds: TgbCustomDataSet; Value: String): String;
function CheckQuotes(Ds: TgbCustomDataSet; Value: String): String;


implementation

uses {$IFDEF D6} Variants,  FMTBcd, {$ENDIF} Forms, gb_dataConsts,
    {$IFDEF VER_IBX} IB; {$ELSE} FIB; {$ENDIF}

var
  IdentChars: set of Char = ['A'..'Z', 'a'..'z', '0'..'9', '_', '-' ,'.', '"', '''' ,'$', #127..#255];

function ExtractIdent(Ds: TgbCustomDataSet; Value: String): String;
begin
  Value := Trim(Value);
  if Ds.Database.SQLDialect = 1 then
    Value := AnsiUpperCase(Value)
  else
  begin
    if (Value <> '') and (Value[1] = '"') then
    begin
      Delete(Value, 1, 1);
      Delete(Value, Length(Value), 1);
      Value := StringReplace (Value, '""', '"', [rfReplaceAll]);
    end
    else
      Value := AnsiUpperCase(Value);
  end;
  Result := Value;
end;

function CheckQuotes(Ds: TgbCustomDataSet; Value: String): String;
begin
  if Ds.Database.SQLDialect = 1 then
    Value := AnsiUpperCase(Trim(Value))
  else
  begin
    if (Length(Value) > 0) then
    begin
      if (Value[1] <> '"') and (Value[Length(Value)] <> '"') then
        Value := '"' + StringReplace (Value, '"', '""', [rfReplaceAll]) + '"';
    end
    else
      Value := '""'
  end;
  Result := Value;
end;

{ TgbCustomDataSet }

constructor TgbCustomDataSet.Create(AOwner: TComponent);
begin
  inherited;
{$IFDEF VER_IBX}
  FGDSLibrary := GetGDSLibrary;
  FGDSLibrary.CheckIBLoaded;
{$ENDIF}
  FBase := {$IFDEF VER_IBX} TIBBase {$ELSE} TFIBBase {$ENDIF}.Create(Self);
  FQSelect := {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}.Create(Self);
  FQSelect.GoToFirstRecordOnExecute := False;
  FQSelectDesc := {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}.Create(Self);
  FQSelectDesc.GoToFirstRecordOnExecute := False;
  FQSelectPart := {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}.Create(Self);
  FQSelectPart.GoToFirstRecordOnExecute := False;
  FQSelectDescPart := {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}.Create(Self);
  FQSelectDescPart.GoToFirstRecordOnExecute := False;
  FTecknics := {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}.Create(Self);
  FTecknics.GoToFirstRecordOnExecute := False;
  FQRefresh := {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}.Create(Self);
  FQRefresh.GoToFirstRecordOnExecute := False;
  FQLocate := {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}.Create(Self);
  FQLocate.GoToFirstRecordOnExecute := False;
  FPostTransaction := {$IFDEF VER_IBX} TIBTransaction {$ELSE} TFIBTransaction {$ENDIF}.Create(Self);
  FPostTransaction.{$IFDEF VER_IBX} Params {$ELSE} TrParams {$ENDIF}.Add('rec_version');
  FPostTransaction.{$IFDEF VER_IBX} Params {$ELSE} TrParams {$ENDIF}.Add('nowait');
  FInsertText := TStringList.Create;
  FDeleteText := TStringList.Create;
  FUpdateText := TStringList.Create;
  FRefreshText := TStringList.Create;
  FKeyFields := TStringList.Create;
  FBase.{$IFDEF VER_IBX} BeforeDatabaseDisconnect {$ELSE} OnDatabaseDisconnecting {$ENDIF}:= DoBeforeDatabaseDisconnect;
  FBase.{$IFDEF VER_IBX} AfterDatabaseDisconnect {$ELSE} OnDatabaseDisconnected {$ENDIF}:= DoAfterDatabaseDisconnect;
  FBase.OnDatabaseFree := DoDatabaseFree;
  FBase.{$IFDEF VER_IBX} BeforeTransactionEnd {$ELSE} OnTransactionEnding {$ENDIF}:= DoBeforeTransactionEnd;
  FBase.{$IFDEF VER_IBX} AfterTransactionEnd {$ELSE} OnTransactionEnded {$ENDIF}:=  DoAfterTransactionEnd;
  FBase.OnTransactionFree := DoTransactionFree;
  FGeneratorDefs := TGeneratorDefs.Create(Self, TGeneratorDef);
  MapOfRecord := TList.Create;
  FWriteBlobs := TList.Create;
  FBufferChunks := 100;
  FSQLWaitCursor := crSQLWait;
  FStreamedActive := False;
  FRefreshWithNextNeed := False;
  FNavOptions := [opCheckNullsInRefreshWhere];
  FUpdateOptions := [opCommitOnPost, opCheckNullsInWhere, opLockRecOnEdit, opWhereChangeOnInsert];
  FReadOnly := False;
  FUseMacros := False;
end;

destructor TgbCustomDataSet.Destroy;
begin
  Close;
  FreeAndNil(FBase);
  FInsertText.Free;
  FDeleteText.Free;
  FUpdateText.Free;
  FRefreshText.Free;
  FKeyFields.Free;
  ClearMapOfRecord;
  MapOfRecord.Free;
  FWriteBlobs.Free;
  CheckCacheState(True);
  FGDSLibrary := nil;
  inherited;
end;

procedure TgbCustomDataSet.ActivateConnection;
begin
  if not Assigned(Database) then
    raise Exception.Create(gb_SNoDatabase);
  if not Assigned(Transaction) then
    raise Exception.Create(gb_SNoTransaction);
  if not Database.Connected then Database.Open;
end;

procedure TgbCustomDataSet.ActivateTransaction;
begin
  if not Assigned(Transaction) then
    raise Exception.Create(gb_SNoTransaction);
  if not Transaction.Active then
    Transaction.StartTransaction;
  if opCommitOnPost in FUpdateOptions then
    if Transaction.{$IFDEF VER_IBX} Params {$ELSE} TrParams {$ENDIF}.IndexOf('read_committed') = -1 then
       raise Exception.Create(gb_SNotCorrectOption);
end;

procedure TgbCustomDataSet.AddToMapOfRecord;
var
  Temp: PFieldInfo;
begin
  New(Temp);
  MapOfRecord.Add(Temp);
end;

function TgbCustomDataSet.AllocRecordBuffer: PChar;
begin
  Result := AllocMem(FRecordBufferSize);
end;

function TgbCustomDataSet.CheckBmkRelative(Bmk: TBookmark; RecNum: integer): Boolean;
var
  x: integer;
  PInfo: PFieldInfo;
begin
  Result := True;
  for x := 0 to FKeyFields.Count - 1 do
  begin
    PInfo := GetMapByName(FKeyFields[x]);
    if not CompareMem(Pchar(FCache[RecNum] + PInfo^.FOffset), PChar(PChar(Bmk) + PInfo^.FBookmarkOffset),
      PInfo^.FDataSize + 1) then
    begin
      Result := False;
      break;
    end;
  end;
end;

procedure TgbCustomDataSet.CheckCacheState(AFree: Boolean);
var
  i: integer;
begin
  if AFree then
  begin
    for i:= Low(FCache) to High(FCache) do
      FreeRecordBuffer(FCache[i]);
    SetLength(FCache, 0);
    FreeMem(TempBuffer);
  end
  else  if  (High(FCache) < 2) or (High(FCache) <> (FBufferChunks - 1))
     or (Integer(StrLen(FCache[0])) <> FRecordBufferSize) then
  begin
    for i:= Low(FCache) to High(FCache) do
      FreeRecordBuffer(FCache[i]);
    if FBufferChunks < 100 then FBufferChunks := 100;
    if Odd(FBufferChunks) then Inc(FBufferChunks);
    SetLength(FCache, FBufferChunks);
    for i:= Low(FCache) to High(FCache) do
    begin
      FCache[i] := AllocRecordBuffer;
      FillChar(FCache[i]^, FRecordBufferSize, 0);
    end;
    TempBuffer := AllocMem((FBufferChunks div 2) * SizeOf(Pointer));
  end;
end;

function TgbCustomDataSet.CheckForNulls(SQL: TStrings;
        CheckType: TUpdateOption; OldBufferNeed: Boolean): string;
var
  Cur, CurEnd, Repl, ReplEnd: integer;
  CurrentToken: string;
  Contin, Blank : Boolean;
  OldBuffer : PChar;
  TempList : TStringList;

  function TokenNext_(const S: string):Boolean;
  begin
    Result := True;
    Cur := CurEnd;
    CurrentToken := TokenNext(S, Blank, Cur, CurEnd);
    if (CurrentToken = '') or (IsStatement(CurrentToken)) then
      Result := False;
  end;

  function ParIsNull(Par: string):Boolean;
  var
    CurBuf: PChar;
    PInfo: PFieldInfo;
  begin
    if Pos('OLD_', Par) = 1 then
    begin
      if OldBufferNeed then
        CurBuf := OldBuffer
      else
        CurBuf := ActiveBuffer;
      Par := Copy(Par, 5, Length(Par));
    end
    else
      CurBuf := ActiveBuffer;
    PInfo := GetMapByName(Par);
    if PInfo <> nil then
      Result := not Boolean(PChar(CurBuf + PInfo^.FOffset)[PInfo^.FDataSize])
    else
      Result := False;
  end;

begin
  Result := SQL.Text;
  Blank := Database.SQLDialect = 3;
  if CheckType = opCheckNullsInWhere then
  begin
    if OldBufferNeed then
    begin
      OldBuffer := AllocRecordBuffer;
      ReadFromCache(FCurrentRecord, OldBuffer);
    end;
    CurEnd := -1;
    Cur := GetPosOfStatement(Result, 'where', CurEnd);
    if Cur > 0 then
    begin
      CurEnd := Cur + 5;
      Contin := TokenNext_(Result);
      while Contin do
      begin
        if CurrentToken = '=' then
        begin
          Repl := Cur;
          Contin := TokenNext_(Result);
          if Contin then
          begin
            if CurrentToken = ':' then
            begin
              Contin := TokenNext_(Result);
              if Contin then
              begin
                if ParIsNull(ExtractIdent(Self, CurrentToken)) then
                begin
                  System.Delete(Result, Repl, CurEnd - Repl);
                  System.Insert(' IS NULL', Result, Repl);
                  CurEnd := Repl + 8;
                end;
                Contin := TokenNext_(Result);
              end;
            end;
          end;
        end
        else
          Contin := TokenNext_(Result);
      end;
    end;
    if OldBufferNeed then FreeRecordBuffer(OldBuffer);
  end
  else
  begin
    TempList := TStringList.Create;
    CurEnd := -1;
    Cur := GetPosOfStatement(Result, 'values', CurEnd);
    if Cur > 0 then
    begin
      CurEnd := Cur + 6;
      TokenNext_(Result);
      Contin := TokenNext_(Result);
      while Contin do
      begin
        if CurrentToken = ':' then
        begin
          Repl := Cur;
          Contin := TokenNext_(Result);
          if Contin then
          begin
            if ParIsNull(ExtractIdent(Self, CurrentToken)) then
            begin
              TempList.Add(ExtractIdent(Self, CurrentToken));
              ReplEnd := CurEnd;
              Contin := TokenNext_(Result);
              if CurrentToken = ',' then
                System.Delete(Result, Repl, CurEnd - Repl)
              else
              begin
                Dec(Repl);
                while (Repl <= Length(Result)) and (Result[Repl]<=' ') do Dec(Repl);
                if Result[Repl] <> ',' then Inc(Repl);
                System.Delete(Result, Repl, ReplEnd - Repl);
              end;
              CurEnd := Repl;
            end;
            if Contin then
              Contin := TokenNext_(Result);
          end;
        end
        else
          Contin := TokenNext_(Result);
      end;
    end;
    if TempList.Count > 0 then
    begin
      CurEnd := -1;
      Cur := GetPosOfStatement(Result, 'into', CurEnd);
      if Cur > 0 then
      begin
        CurEnd := Cur + 6;
        TokenNext_(Result); TokenNext_(Result);
        Contin := TokenNext_(Result);
        while Contin do
        begin
          if (CurrentToken <> ',') and (TempList.IndexOf(ExtractIdent(Self, CheckRelation(CurrentToken, False))) <> -1) then
          begin
            Repl := Cur; ReplEnd := CurEnd;
            Contin := TokenNext_(Result);
            if CurrentToken = ',' then
              System.Delete(Result, Repl, CurEnd - Repl)
            else
            begin
              Dec(Repl);
              while (Repl <= Length(Result)) and (Result[Repl]<=' ') do Dec(Repl);
              if Result[Repl] <> ',' then Inc(Repl);
              System.Delete(Result, Repl, ReplEnd - Repl);
            end;
            CurEnd := Repl;
          end;
          if Contin then
            Contin := TokenNext_(Result);
        end;
      end;
    end;
    TempList.Free;
  end;
end;

procedure TgbCustomDataSet.CheckGenerators(GenType: TGeneratorTypes);
const
  SGENSQL = 'SELECT GEN_ID(%s, %d) FROM RDB$DATABASE';
var
  x: integer;
  TempIBSQL: {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF};
begin
  TempIBSQL := nil;
  try
    for x := 0 to GeneratorDefs.Count - 1 do
    begin
      if GeneratorDefs[x].FGeneratorType = gtServer then
      begin
        if FieldByName(GeneratorDefs[x].FFieldName).Required then
            FieldByName(GeneratorDefs[x].FFieldName).Required := False;
      end
      else
        if (GeneratorDefs[x].FGeneratorType in GenType) and (GeneratorDefs[x].FFieldName <> '') and
          (GeneratorDefs[x].FGeneratorName <> '') then
        begin
          if TempIBSQL = nil then
          begin
            TempIBSQL := {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}.Create(Database);
            TempIBSQL.Transaction := Transaction;
          end;
          if TempIBSQL.Open then TempIBSQL.Close;
          TempIBSQL.SQL.Text := Format(SGENSQL, [CheckQuotes(Self, GeneratorDefs[x].FGeneratorName), GeneratorDefs[x].FIncrementBy]);
          TempIBSQL.ExecQuery;
          if FieldByName(GeneratorDefs[x].FFieldName).ClassType <> TLargeIntField then
            FieldByName(GeneratorDefs[x].FFieldName).AsInteger := TempIBSQL.Current.Vars[0].AsInt64
          else
            TLargeIntField(FieldByName(GeneratorDefs[x].FFieldName)).AsLargeInt := TempIBSQL.Current.Vars[0].AsInt64;
        end;
    end;
  finally
    if TempIBSQL <> nil then
      TempIBSQL.Free;
  end;
end;

function TgbCustomDataSet.CheckNext(Buffer: PChar): TGetResult;
begin
  Result := grOK;
  if FCurrentRecord = (FRecordCount - 1) then
  begin
    if not FLastIsReal then
    begin
      if DescNow then
      begin
        if FQSelectPart.Open then FQSelectPart.Close;
        SetParamsTo(FQSelectPart, nil, nil, False);
        ExecWithCursor(FQSelectPart);
        CurrentSelect := FQSelectPart;
        if (FRecordCount < FBufferChunks) and (FAboveDesc > 0) then
           CorrectTop(FAboveDesc);
      end;
      if FCurrentRecord = (FBufferChunks - 1) then
          ReorderCache(True);
    end
    else
      Result := grEOF;
  end;
end;

function TgbCustomDataSet.CheckPrior(Buffer: PChar): TGetResult;
begin
  Result := grOK;
  if FCurrentRecord = 0 then
  begin
    ReorderCache(False);
    FAboveDesc := FAboveDesc + (FBufferChunks div 2);
    if not DescNow then
    begin
      if FQSelectDescPart.Open then FQSelectDescPart.Close;
      SetParamsTo(FQSelectDescPart, nil, nil, False);
      ExecWithCursor(FQSelectDescPart);
      CurrentSelect := FQSelectDescPart;
    end;
    if CurrentSelect.Next <> nil then
    begin
      Dec(FCurrentRecord);
      FetchRecordToBuffer(CurrentSelect, FCurrentRecord, Buffer);
      FAboveDesc := FCurrentRecord;
      Result := grOK;
    end
    else
    begin
      First;
      Result := grBOF;
    end;
  end;
end;

function TgbCustomDataSet.CheckRelation(const S: string; ACheck: Boolean): string;
var
  p, i: integer;
  Str: string;
begin
  Result := S;
  p := Pos('.', S);
  if (p > 0) or ACheck then
  begin
    if (p > 0) then
      Str := ExtractIdent(Self, Copy(S, 1, p - 1)) + '.' + ExtractIdent(Self, Copy(S, p + 1, Length(S)))
    else
      Str := ExtractIdent(Self, S);
    for i := 0 to MapOfRecord.Count - 1 do
    begin
      if (p > 0) then
      begin
        if AnsiSameText(Str, PFieldInfo(MapOfRecord[i])^.FRelationNameForOrder) then
        begin
          Result := PFieldInfo(MapOfRecord[i])^.FFieldName;
          Exit;
        end
        else
          if (Length(PFieldInfo(MapOfRecord[i])^.FRelationNameForOrder) > 0) and
             (PFieldInfo(MapOfRecord[i])^.FRelationNameForOrder[1] = '.') then
          begin
            if AnsiSameText(Copy(Str, (Pos('.', Str) + 1), Length(Str)), PFieldInfo(MapOfRecord[i])^.FFieldName) then
            begin
              Result := PFieldInfo(MapOfRecord[i])^.FFieldName;
              Exit;
            end;
          end;
      end
      else
        if AnsiSameText(Str, PFieldInfo(MapOfRecord[i])^.FFieldName) then Exit;
    end;
    raise Exception.Create(gb_SNoFullName);
  end;
end;

procedure TgbCustomDataSet.ClearCalcFields(Buffer: PChar);
begin
  FillChar(Buffer[FCalcFieldsOffset], CalcFieldsSize, 0);
end;

procedure TgbCustomDataSet.ClearMapOfRecord;
var
  x: integer;
begin
  for x := 0 to MapOfRecord.Count - 1 do
    Dispose(PFieldInfo(MapOfRecord[x]));
  MapOfRecord.Clear;
end;

{$IFDEF VER_FIBPLUS}
procedure TgbCustomDataSet.CloseSelectQuery;
begin
  if FQSelect.Open then FQSelect.Close;
end;
{$ENDIF}

function TgbCustomDataSet.CompareBookmarks(Bookmark1, Bookmark2: TBookmark): Integer;
var
  x: integer;
begin
  Result := 0;
  if Assigned(Bookmark1) and Assigned(Bookmark2) then
  begin
    for x := SizeOf(Integer) to BookmarkSize - 1 do
    begin
      if not CompareMem(PChar(Bookmark1) + x, PChar(Bookmark2) + x, 1) then
      begin
        if PInteger(Bookmark1)^ <> PInteger(Bookmark2)^ then
        begin
          if PInteger(Bookmark1)^ < PInteger(Bookmark2)^ then
            Result := -1
          else
            Result :=  1;
        end
        else
        begin
          if Ord(PChar(Bookmark1)[x]) < Ord(PChar(Bookmark2)[x]) then
            Result := -1
          else
            Result :=  1;
        end;
      end;
    end;
  end;
end;

procedure TgbCustomDataSet.ConvertSQLs(ReAssign: Boolean);
var
  SList : TStringList;
  WhereStr : string;

  function SetLockSQLText: string;
  var
   x, y: integer;
  begin
    Result := '';
    y := -1;
    x := GetPosOfStatement(UpdateSQL.Text, 'set', y);
    if x > 0 then
    begin
      Result := Copy(UpdateSQL.Text, 1, x + 2) + '  ';
      Result := Result + ' ' +  CheckQuotes(Self, FKeyFields[0]) + ' =:' + CheckQuotes(Self, CheckRelation(FKeyFields[0], False)) + ' where ';
      for x := 0 to FKeyFields.Count - 1 do
        if x = FKeyFields.Count - 1 then
          Result := Result + ' ' +  CheckQuotes(Self, FKeyFields[x]) + ' =:' + CheckQuotes(Self, CheckRelation(FKeyFields[x], False))
        else
          Result := Result + ' ' +  CheckQuotes(Self, FKeyFields[x]) + ' =:' + CheckQuotes(Self, CheckRelation(FKeyFields[x], False)) + ' AND ';
    end;
  end;

begin
{$IFDEF VER_FIBPLUS}
   if FQSelectPart.Open then FQSelectPart.Close;
   if FQSelectDesc.Open then FQSelectDesc.Close;
   if FQSelectDescPart.Open then FQSelectDescPart.Close;
{$ENDIF}
   if ReAssign then
   begin
     FQSelectPart.SQL.Assign(FQSelect.SQL);
     FQSelectDesc.SQL.Assign(FQSelect.SQL);
     FQSelectDescPart.SQL.Assign(FQSelect.SQL);
   end;
   SList := TStringList.Create;
   try
    GetFieldsInOrder_(FQSelect.SQL, SList);
    if SList.Count = 0 then
      raise Exception.Create(gb_SNoOrderBy);
    OrderToDesc(FQSelectDesc.SQL, SList);
    WhereStr := GenerateWhere(SList, True);
    InsertWhere(WhereStr, FQSelectPart.SQL);
    OrderToDesc(FQSelectDescPart.SQL, SList);
    WhereStr := GenerateWhere(SList, False);
    InsertWhere(WhereStr, FQSelectDescPart.SQL);
    if UpdateSQL.Text <> '' then  FLockSQLText := SetLockSQLText;
   finally
    SList.Free;
   end;
end;

procedure TgbCustomDataSet.CorrectTop(Above: integer);
var
  Buf, Buf2: PChar;
  CountByte1, CountByte2, x: integer;
begin
  CountByte1 := SizeOf(Pointer) * Above;
  CountByte2 := SizeOf(Pointer) * (FRecordCount - Above);
  Buf := AllocMem(CountByte1);
  Move(FCache[0], Buf^, CountByte1);
  Move(FCache[Above], FCache[0], CountByte2);
  Move(Buf^, FCache[FRecordCount - Above], CountByte1);
  FRecordCount := FRecordCount - Above;
  FCurrentRecord := FCurrentRecord - Above;
  for x := 0 to FRecordCount - 1 do
  begin
    Buf2 := FCache[x];
    PRecInfo(Buf2 + FRecordSize)^.Bookmark := PRecInfo(Buf2 + FRecordSize)^.Bookmark - Above;
  end;
  for x := 0 to BufferCount - 1 do
  begin
    Buf2 := Buffers[x];
    PRecInfo(Buf2 + FRecordSize)^.Bookmark := PRecInfo(Buf2 + FRecordSize)^.Bookmark - Above;
  end;
  FreeMem(Buf);
end;

function TgbCustomDataSet.DescNow: Boolean;
begin
  Result := (CurrentSelect = FQSelectDesc) or (CurrentSelect = FQSelectDescPart);
end;

procedure TgbCustomDataSet.DoAfterDatabaseDisconnect(Sender: TObject);
begin
  if Assigned(FAfterDatabaseDisconnect) then
    FAfterDatabaseDisconnect(Sender);
end;

procedure TgbCustomDataSet.DoAfterPost;
begin
  if FRefreshWithNextNeed then
  try
    Next;
    Refresh;
  finally
    FRefreshWithNextNeed := False;
  end;
  inherited;
end;

procedure TgbCustomDataSet.DoAfterTransactionEnd(Sender: TObject);
begin
  if Assigned(FAfterTransactionEnd) then
    FAfterTransactionEnd(Sender);
end;

procedure TgbCustomDataSet.DoBeforeDatabaseDisconnect(Sender: TObject);
begin
  if Active then Close;
  if Assigned(FBeforeDatabaseDisconnect) then
    FBeforeDatabaseDisconnect(Sender);
end;

procedure TgbCustomDataSet.DoBeforePost;
begin
  FRefreshWithNextNeed := False;
  case State of
    dsInsert : CheckGenerators([gtOnInsert, gtOnInsertEdit]);
    dsEdit   : CheckGenerators([gtOnEdit, gtOnInsertEdit]);
  end;
  inherited;
end;

procedure TgbCustomDataSet.DoBeforeTransactionEnd(Sender: TObject);
begin
  if Active then Close;
  if FQSelect <> nil then
    FQSelect.FreeHandle;
  if FQSelectDesc <> nil then
    FQSelectDesc.FreeHandle;
  if FQSelectDescPart <> nil then
    FQSelectDescPart.FreeHandle;
  if FQSelectPart <> nil then
    FQSelectPart.FreeHandle;
  if FQRefresh <> nil then
    FQRefresh.FreeHandle;
  if FQLocate <> nil then
    FQLocate.FreeHandle;
  if FTecknics <> nil then
    FTecknics.FreeHandle;
  if FPostTransaction.Active then
    FPostTransaction.Rollback;
  if Assigned(FBeforeTransactionEnd) then
    FBeforeTransactionEnd(Sender);
end;

procedure TgbCustomDataSet.DoDatabaseFree(Sender: TObject);
begin
  if Assigned(FDatabaseFree) then
    FDatabaseFree(Sender);
end;

procedure TgbCustomDataSet.DoOnNewRecord;
begin
  CheckGenerators([gtOnNewRecord]);
  inherited;
end;

procedure TgbCustomDataSet.DoTransactionFree(Sender: TObject);
begin
  if Assigned(FTransactionFree) then
    FTransactionFree(Sender);
end;

procedure TgbCustomDataSet.ExecWithCursor(IBSQL: {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF});
var
  CurChanged: Boolean;
  OldCursor: TCursor;
begin
  OldCursor := Screen.Cursor;
  if Screen.Cursor <> FSQLWaitCursor then
  begin
    Screen.Cursor := FSQLWaitCursor;
    CurChanged := True;
  end
  else
    CurChanged := False;
  try
    IBSQL.ExecQuery;
  finally
    if CurChanged then
      Screen.Cursor := OldCursor;
  end;
end;

procedure TgbCustomDataSet.FetchRecordToBuffer(Qry: {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}; ARecNo: integer; Buffer: PChar);
var
  i: Integer;
  LocalData: Pointer;
  LocalDouble: Double;
  LocalInt: Integer;
  LocalInt64: Int64;
  LocalCurrency: Currency;
  TekLen, LocalSmall: SmallInt;
  PInfo: PFieldInfo;
begin
  for i := 0 to Qry.Current.Count - 1 do
  begin
    PInfo := GetMap(i, False);
    if PInfo^.FUnknownType then Continue;

{$IFNDEF VER_IBX}
    if not ((Qry.Current[i].Data^.sqltype and 1 = 1) and
                               (Qry.Current[i].Data^.sqlind^ = -1)) then
{$ELSE}
    if not ((Qry.Current[i].Data.sqltype and 1 = 1) and
                               (Qry.Current[i].Data.sqlind^ = -1)) then
{$ENDIF}
    begin
{$IFNDEF VER_IBX}
      LocalData := Qry.Current[i].Data^.sqldata;
{$ELSE}
      LocalData := Qry.Current[i].Data.sqldata;
{$ENDIF}
      TekLen := 0;
      case PInfo^.FType of
        SQL_TIMESTAMP:
        begin
          LocalDouble := TimeStampToMSecs(DateTimeToTimeStamp(Qry.Current[i].AsDateTime));
          LocalData := PChar(@LocalDouble);
        end;
        SQL_TYPE_DATE:
        begin
          LocalInt := DateTimeToTimeStamp(Qry.Current[i].AsDateTime).Date;
          LocalData := PChar(@LocalInt);
        end;
        SQL_TYPE_TIME:
        begin
          LocalInt := DateTimeToTimeStamp(Qry.Current[i].AsDateTime).Time;
          LocalData := PChar(@LocalInt);
        end;
        SQL_SHORT, SQL_LONG:
        begin
          if (PInfo^.FScale = 0) then
          begin
            if PInfo^.FType =  SQL_LONG then
            begin
              LocalInt := Qry.Current[i].AsLong;
              LocalData := PChar(@LocalInt);
            end
            else
            begin
              LocalSmall := Qry.Current[i].AsShort;
              LocalData := PChar(@LocalSmall);
            end;
          end
          else if (PInfo^.FScale >= (-4)) then
               begin
                 LocalCurrency := Qry.Current[i].AsCurrency;
                 LocalData := PChar(@LocalCurrency);
               end
               else begin
                LocalDouble := Qry.Current[i].AsDouble;
                LocalData := PChar(@LocalDouble);
              end;
        end;
        SQL_INT64:
        begin
          if (PInfo^.FScale = 0) then
          begin
            LocalInt64 := Qry.Current[i].AsInt64;
            LocalData := PChar(@LocalInt64);
          end
          else
            if (PInfo^.FScale >= (-4)) then
            begin
              LocalCurrency := Qry.Current[i].AsCurrency;
              LocalData := PChar(@LocalCurrency);
            end
            else
            begin
              LocalDouble := Qry.Current[i].AsDouble;
              LocalData := PChar(@LocalDouble);
            end
        end;
        SQL_DOUBLE, SQL_FLOAT, SQL_D_FLOAT:
        begin
          LocalDouble := Qry.Current[i].AsDouble;
          LocalData := PChar(@LocalDouble);
        end;
        SQL_VARYING:
        begin
{$IFNDEF VER_IBX}
          TekLen := FGDSLibrary.isc_vax_integer(Qry.Current[i].Data^.sqldata, 2);
{$ELSE}
          TekLen := FGDSLibrary.isc_vax_integer(Qry.Current[i].Data.sqldata, 2);
{$ENDIF}
          if (TekLen = 0) then
            LocalData := nil
          else
{$IFNDEF VER_IBX}
            LocalData := @Qry.Current[i].Data^.sqldata[2];
{$ELSE}
            LocalData := @Qry.Current[i].Data.sqldata[2];
{$ENDIF}
        end;
        else
        begin
          if (PInfo^.FType = SQL_TEXT) then
{$IFNDEF VER_IBX}
            TekLen := Qry.Current[i].Data^.sqllen;
{$ELSE}
            TekLen := Qry.Current[i].Data.sqllen;
{$ENDIF}
        end;
      end;
        if (PInfo^.FType = SQL_VARYING) or (PInfo^.FType = SQL_TEXT) then
        begin
          if LocalData <> nil then
          begin
            Move(LocalData^, Buffer[PInfo^.FOffset], TekLen);
            if TekLen < PInfo^.FDataSize then
              Buffer[PInfo^.FOffset + TekLen] := #0;
          end
          else
            Buffer[PInfo^.FOffset] := #0;
        end
        else
          Move(LocalData^, Buffer[PInfo^.FOffset], PInfo^.FDataSize);

       Boolean(Buffer[PInfo^.FOffset + PInfo^.FDataSize]) := Boolean(True);
    end
    else
      Boolean(Buffer[PInfo^.FOffset + PInfo^.FDataSize]) := Boolean(False);
  end;
  PRecInfo(Buffer + FRecordSize).Bookmark := ARecNo;
  if ARecNo >= 0 then
    WriteToCache(ARecNo, Buffer);
end;

procedure TgbCustomDataSet.FreeRecordBuffer(var Buffer: PChar);
begin
   FreeMem(Buffer);
end;

function TgbCustomDataSet.GenerateWhere(SList: TStrings; NoDesc: Boolean):string;
var
  x, y: integer;
  Sign: char;
  S, T, Par: string;
begin
  Result := '';
  for x := 0 to SList.Count - 1 do
  begin
    S := '';
    T := '';
    for y := 0 to SList.Count - x - 1 do
    begin
      Par := CheckRelation(SList[y], False);
      if Integer(SList.Objects[y]) = 0 then
        if NoDesc then Sign := '>' else Sign := '<'
      else
        if NoDesc then Sign := '<' else Sign := '>';

      if y < (SList.Count - x - 1) then
         T := SList[y] + ' = :' + CheckQuotes(Self, Par) + ' AND'
      else
      begin
         T := SList[y] + ' '+ Sign +' :' + CheckQuotes(Self, Par);
      end;
      S := S + ' ' + T;
    end;
    S := '(' + S + ')';
    if x < SList.Count - 1 then
       Result := Result + S + ' OR '
    else
       Result := Result + S;
  end;
  if SList.Count > 1 then
    Result := '(' + Result + ')';
end;

function TgbCustomDataSet.GetActiveBuf: PChar;
begin
  case State of
   dsBrowse:
     if IsEmpty then
       result := nil
     else
       result := ActiveBuffer;
   dsEdit, dsInsert:
     result := ActiveBuffer;
   dsCalcFields:
     result := CalcBuffer;
  else
    if not FOpen then
      result := nil
    else
      result := ActiveBuffer;
  end;
end;

procedure TgbCustomDataSet.GetBookmarkData(Buffer: PChar; Data: Pointer);
var
 x: integer;
 PInfo: PFieldInfo;
begin
  PInteger(Data)^ := PRecInfo(Buffer + FRecordSize).Bookmark;
  for x := 0 to FKeyFields.Count - 1 do
  begin
    PInfo := GetMapByName(FKeyFields[x]);
    Move(PChar(Buffer + PInfo^.FOffset)^, PChar(PChar(Data) + PInfo^.FBookmarkOffset)^, PInfo^.FDataSize + 1);
  end;
end;

function TgbCustomDataSet.GetBookmarkFlag(Buffer: PChar): TBookmarkFlag;
begin
  Result := PRecInfo(Buffer + FRecordSize).BookmarkFlag;
end;

function TgbCustomDataSet.GetCanModify: Boolean;
begin
  Result := not FReadOnly;
end;

function TgbCustomDataSet.GetDatabase: {$IFDEF VER_IBX} TIBDatabase {$ELSE} TFIBDatabase {$ENDIF};
begin
  Result := FBase.Database;
end;

function TgbCustomDataSet.GetDeleteSQL: TStrings;
begin
  if not (FUseMacros and Active) then
    Result := FDeleteText
  else
    Result := SynchronizeSQL(3);
end;

function TgbCustomDataSet.GetFieldData(Field: TField; Buffer: Pointer): Boolean;
var
  PInfo: PFieldInfo;
  lTempCurr: System.Currency;
  CurBuf: PChar;
  Len: integer;
begin
  CurBuf := GetActiveBuf;
  if CurBuf = nil then
  begin
    Result := False;
    Exit;
  end;
  if Field.FieldNo > 0 then
  begin
    PInfo := GetMap(Field.FieldNo - 1, True);
    Result := Boolean(CurBuf[PInfo^.FOffset + PInfo^.FDataSize]);
    if Result and Assigned(Buffer) then
    begin
      if (Field.DataType = ftBCD) then
      begin
        Move( PChar(CurBuf + PInfo^.FOffset)^ , PChar(@lTempCurr)^, PInfo^.FDataSize);
        CurrToBCD(lTempCurr, TBCD(Buffer^), 32, Field.Size);
      end
      else
        if (PInfo^.FType = SQL_TEXT) and (Field is TStringField) and
         ((TStringField(Field).EditMask <> '') or (not TStringField(Field).FixedChar)) then
        begin
          Len := StrLen(PChar(TrimRight(PChar(CurBuf + PInfo^.FOffset))));
          Move( PChar(TrimRight(PChar(CurBuf + PInfo^.FOffset)))^  , Buffer^, Len);
          if Len < PInfo^.FDataSize then
            PChar(Buffer)[Len] := #0;
        end
        else
          Move( PChar(CurBuf + PInfo^.FOffset)^ , Buffer^, PInfo^.FDataSize);
    end;
  end
  else
  begin
   Inc(CurBuf, FCalcFieldsOffset + Field.Offset);
   Result := Boolean(CurBuf[0]);
   if Result and (Buffer <> nil) then
       Move(CurBuf[1], Buffer^, Field.DataSize);
  end;
end;

function TgbCustomDataSet.GetFieldData(Field: TField; Buffer: Pointer; NativeFormat: Boolean): Boolean;
var
  PInfo: PFieldInfo;
  CurBuf: PChar;
begin
  if (Field.DataType = ftBCD) and (not NativeFormat) and (Field.FieldNo > 0) then
  begin
    CurBuf := GetActiveBuf;
    if CurBuf = nil then
    begin
      Result := False;
      Exit;
    end;
    PInfo := GetMap(Field.FieldNo - 1, True);
    Result := Boolean(CurBuf[PInfo^.FOffset + PInfo^.FDataSize]) ;
    if Result and Assigned(Buffer) then
      Move( PChar(CurBuf + PInfo^.FOffset)^ , PChar(Buffer)^, PInfo^.FDataSize);
  end
  else
    Result := inherited GetFieldData(Field, Buffer, NativeFormat);
end;

procedure TgbCustomDataSet.GetFieldsInOrder(Fields: TStrings);
begin
  GetFieldsInOrder_(FQSelect.SQL, Fields);
end;

procedure TgbCustomDataSet.GetFieldsInOrder_(SQL, SList: TStrings);
var
  CurPos, CurEnd: integer;
  S, CurToken : string;
  Blank: Boolean;
begin
  S := SQL.Text;
  CurEnd := -1;
  Blank := Database.SQLDialect = 3;
  CurPos := GetPosOfStatement(S, 'order', CurEnd);
  if CurPos > 0 then
  begin
    while (CurPos <= Length(S)) and (AnsiUpperCase(S[CurPos]) <> 'B') do Inc (CurPos);
    CurPos := CurPos + 2;
    CurToken := TokenNext(S, Blank, CurPos, CurEnd);
    while (CurToken <> '')  and ( not IsStatement(CurToken)) do
    begin
      {if CompareText(CurToken, 'DESC') = 0  then
         raise Exception.Create(gb_SDescNotSupported);}

      if (CurToken <> ',') and (CompareText(CurToken, 'ASC') <> 0)
        and (CompareText(CurToken, 'ascending') <> 0) then
      begin
        if (CompareText(CurToken, 'DESC') = 0)
         or (CompareText(CurToken, 'descending') = 0) then
          SList.Objects[SList.Count - 1] := Pointer(1)
        else
          SList.AddObject(CurToken, Pointer(0));
      end;

      CurPos := CurEnd;
      CurToken := TokenNext(S, Blank, CurPos, CurEnd);
    end;
  end;
end;

function TgbCustomDataSet.GetInsertSQL: TStrings;
begin
  if not (FUseMacros and Active) then
    Result := FInsertText
  else
    Result := SynchronizeSQL(1);
end;

function TgbCustomDataSet.GetMap(I: integer; ACheck: Boolean): PFieldInfo;
begin
  Result := PFieldInfo(MapOfRecord[I]);
  if ACheck then Result := PFieldInfo(MapOfRecord[I + Result^.FHoleNumber]);
end;

function TgbCustomDataSet.GetMapByName(const FieldName: string): PFieldInfo;
var
  I: integer;
begin
  Result := nil;
  for I := 0 to MapOfRecord.Count - 1 do
  begin
    if AnsiUpperCase(PFieldInfo(MapOfRecord[I])^.FFieldName) = AnsiUpperCase(FieldName) then
    begin
      Result := PFieldInfo(MapOfRecord[I]);
      break;
    end;
  end;
end;

function TgbCustomDataSet.GetMapCount: integer;
begin
  Result := MapOfRecord.Count;
end;

function TgbCustomDataSet.GetPosOfStatement(const SQLText, Statement: string; var NextPos: integer): integer;
var
  CurPos, CurEnd, BracketCount: integer;
  CurrentToken: string;
  BlankInt, NextNow: Boolean;

  function CheckForBy: Boolean;
  var
   x: integer;
  begin
    Result := False;
    if (AnsiUpperCase(CurrentToken) = 'ORDER') or
      (AnsiUpperCase(CurrentToken) = 'GROUP') then
    begin
      x := CurPos + 5;
      while (x <= Length(SQLText)) and (SQLText[x] <= ' ') do Inc(x);
      if (x < Length(SQLText) - 1) and (AnsiUpperCase(SQLText[x]) = 'B') and
        (AnsiUpperCase(SQLText[x + 1]) = 'Y') and (AnsiUpperCase(SQLText[x + 2]) <= ' ') then
          Result := True;
    end
    else
      Result := True;
  end;

begin
  Result := -1;
  if Length(SQLText) > 0 then CurPos := 1 else Exit;
  BlankInt := Database.SQLDialect = 3;
  NextNow := False;
  CurrentToken := TokenNext(SQLText, BlankInt, CurPos, CurEnd);
  while CurrentToken <> '' do
  begin
    if NextNow then
    begin
      if IsStatement(CurrentToken) and CheckForBy then
      begin
        NextPos := CurPos;
        Exit;
      end;
    end
    else
    begin
      if CompareText(CurrentToken, Statement) = 0 then
      begin
        if CheckForBy then
        begin
          Result := CurPos;
          if NextPos < 0 then
            Exit
          else
            NextNow := True;
        end;
      end;
    end;
    if CurrentToken = '(' then
    begin
      BracketCount := 1;
      while BracketCount > 0 do
      begin
        CurPos := CurEnd;
        CurrentToken := TokenNext(SQLText, BlankInt, CurPos, CurEnd);
        if CurrentToken = ')' then
          Dec(BracketCount)
        else
          if CurrentToken = '(' then Inc(BracketCount);
      end;
    end
    else
    begin
      CurPos := CurEnd;
      CurrentToken := TokenNext(SQLText, BlankInt, CurPos, CurEnd);
    end;
  end;
  if NextNow then NextPos := CurPos;
end;

{$IFDEF VER_FIBPLUS}
function TgbCustomDataSet.GetQSelect: TFIBQuery;
begin
  Result := FQSelect;
end;
{$ENDIF}

{$IFDEF VER_IBX}
function TgbCustomDataSet.GetQSelect: TIBSQL;
begin
  Result := FQSelect;
end;
{$ENDIF}

function TgbCustomDataSet.GetRecord(Buffer: PChar; GetMode: TGetMode;
  DoCheck: Boolean): TGetResult;
begin
  Result := grError;
  case GetMode of
   gmCurrent:
    begin
     if (FCurrentRecord >= 0) then
     begin
       if FCurrentRecord < FRecordCount then
       begin
         ReadFromCache(FCurrentRecord, Buffer);
         Result := grOk;
       end
       else
       begin
         Result := grEOF;
       end;
     end
     else
       Result := grBOF;
    end;
   gmNext:
    begin
     Result := grOk;
     if FCurrentRecord = FRecordCount then
     begin
       Result := grEOF;
       FLastIsReal := True;
     end
     else
     if FCurrentRecord = FRecordCount - 1 then
     begin
       Result :=  CheckNext(Buffer);
       if Result <> grEOF then
       begin
         if (not CurrentSelect.EOF) then
         begin
           CurrentSelect.Next;
           Inc(FCurrentRecord);
         end;
         if (CurrentSelect.EOF) then
         begin
           Result := grEOF;
           FLastIsReal := True;
         end
         else
         begin
           Inc(FRecordCount);
           FetchRecordToBuffer(CurrentSelect, FCurrentRecord, Buffer);
         end;
       end;
     end
     else
       if (FCurrentRecord < FRecordCount) then
       begin
         Inc(FCurrentRecord);
         ReadFromCache(FCurrentRecord, Buffer);
       end;
   end;
  gmPrior:
   begin
     if (FCurrentRecord = 0) then
     begin
       if FFirstIsReal then
       begin
         Dec(FCurrentRecord);
         Result := grBOF;
       end
       else
       begin
         Result := CheckPrior(Buffer);
       end;
     end
     else
       if (FCurrentRecord > 0) and
                   (FCurrentRecord <= FRecordCount) then
       begin
         Dec(FCurrentRecord);
         if DescNow and (FAboveDesc > FCurrentRecord) then
         begin
           if CurrentSelect.Next <> nil then
            begin
              FetchRecordToBuffer(CurrentSelect, FCurrentRecord, Buffer);
              FAboveDesc := FCurrentRecord;
            end
            else
            begin
              if not FLastIsReal then First;
              Result := grBOF;
            end;
         end
         else
           ReadFromCache(FCurrentRecord, Buffer);
         if (Result <> grBOF) and (Result <> grEOF) then Result := grOk;
       end
       else
         if (FCurrentRecord = -1) then
         begin
           Result := grBOF;
           FFirstIsReal := True;
         end;
   end;
  end;
   if Result = grOk then
   begin
     PRecInfo(Buffer + FRecordSize).BookmarkFlag := bfCurrent;
     GetCalcFields(Buffer);
   end
   else
     if (Result = grEOF) or (Result = grError) then
     begin
       FillChar(Buffer^, FRecordBufferSize, 0);
       PRecInfo(Buffer + FRecordSize).BookmarkFlag := bfEOF;
     end
     else
       if Result = grBOF then
       begin
         FillChar(Buffer^, FRecordBufferSize, 0);
         PRecInfo(Buffer + FRecordSize).BookmarkFlag := bfBOF;
       end;
end;

function TgbCustomDataSet.GetRecordSize: Word;
begin
  Result := FRecordBufferSize;
end;

function TgbCustomDataSet.GetRefreshSQL: TStrings;
begin
  if not (FUseMacros and Active) then
    Result := FRefreshText
  else
    Result := SynchronizeSQL(4);
end;

function TgbCustomDataSet.GetSelectSQL: TStrings;
begin
  if not (FUseMacros and Active) then
    Result := FQSelect.SQL
  else
    Result := SynchronizeSQL(0);
end;

function TgbCustomDataSet.GetTransaction: {$IFDEF VER_IBX} TIBTransaction {$ELSE} TFIBTransaction {$ENDIF};
begin
  Result := FBase.Transaction;
end;

function TgbCustomDataSet.GetUpdateSQL: TStrings;
begin
  if not (FUseMacros and Active) then
    Result := FUpdateText
  else
    Result := SynchronizeSQL(2);
end;

procedure TgbCustomDataSet.InsertWhere(const S: string; SQL: TStrings);
var
  CurPos, ANext: integer;
  Buf: string;
begin
  Buf := SQL.Text;
  ANext := -1;
  CurPos := GetPosOfStatement(Buf, 'where', ANext);
  if CurPos > 0 then
    System.Insert(' ' + S +' AND ', Buf, CurPos + 5)
  else
  begin
    ANext := 0;
    CurPos := GetPosOfStatement(Buf, 'from', ANext);
    if CurPos > 0 then
      System.Insert(' where ' + S + ' ' , Buf, ANext - 1);
  end;
  SQL.Text := Buf;
end;

procedure TgbCustomDataSet.InternalCancel;
begin
  ClearWriteBlobs;
  if FPostTransaction.Active then
    FPostTransaction.Rollback;
//  if (State = dsEdit) and (opLockRecOnEdit in FUpdateOptions) then
//    RefreshCurrentRecord(False);
  inherited;
end;

procedure TgbCustomDataSet.InternalClose;
begin
  FQSelect.Close;
  FQSelectPart.Close;
  FQSelectDescPart.Close;
  FQSelectDesc.Close;
  FTecknics.Close;
  FQRefresh.Close;
  FQLocate.Close;
  ClearWriteBlobs;
  FOpen := False;
  BindFields(False);
  if DefaultFields then
    DestroyFields;
{$IFNDEF VER_IBX}
  FGDSLibrary := nil;
{$ENDIF}
end;

procedure TgbCustomDataSet.InternalDelete;
var
  NotDelSucs: Boolean;
begin
  if opCheckNullsInWhere in FUpdateOptions then
    FTecknics.SQL.Text := CheckForNulls(FDeleteText, opCheckNullsInWhere, False)
  else
    FTecknics.SQL.Assign(FDeleteText);
  if (opCommitOnPost in FUpdateOptions) and (not FPostTransaction.Active) then
    FPostTransaction.StartTransaction;
  try
    SetParamsTo(FTecknics, nil, nil, False);
    ExecWithCursor(FTecknics);
  except
    on E: {$IFDEF VER_IBX} EIBError {$ELSE} EFIBError {$ENDIF} do
    begin
      if FPostTransaction.Active then
        FPostTransaction.Rollback;
      if E.SQLCode = - 901 then
        raise EDatabaseError.Create(gb_SRecordIsLocked)
      else
        raise;
    end;
    else
    begin
      if FPostTransaction.Active then
        FPostTransaction.Rollback;
      raise;
    end;
  end;

  NotDelSucs := False;
  try
   if FTecknics.RowsAffected > 0 then
   begin
     MoveInCache(FCurrentRecord, FRecordCount - 1);
     Dec(FRecordCount);
   end
   else
   begin
     NotDelSucs := True;
     raise EDatabaseError.Create(gb_SRecWasModified);
   end;
  finally
   if opCommitOnPost in FUpdateOptions then
   if NotDelSucs then
   begin
     FPostTransaction.Rollback;
     Refresh;
   end
   else
     FPostTransaction.Commit;
  end;
end;

procedure TgbCustomDataSet.InternalEdit;
begin
  if (opLockRecOnEdit in FUpdateOptions) and (FLockSQLText <> '') then
  begin
    if opCommitOnPost in FUpdateOptions then
      FPostTransaction.StartTransaction;
    if LockCurrentRecord then
      RefreshCurrentRecord(opCommitOnPost in FUpdateOptions)
    else
    begin
      if FPostTransaction.Active then
        FPostTransaction.Rollback;
      Refresh;
      raise EDatabaseError.Create(gb_SRecWasModified);
    end;
  end
  else
    RefreshCurrentRecord(False);
  inherited;
end;

procedure TgbCustomDataSet.InternalFirst;
begin
  if not FFirstIsReal then
  begin
    if FQSelect.Open then FQSelect.Close;
    SetParamsTo(FQSelect, nil, nil, False);
    ExecWithCursor(FQSelect);
    FFirstIsReal := True;
    FLastIsReal := False;
    CurrentSelect := FQSelect;
    FRecordCount := 0;
  end;
  FCurrentRecord := -1;
end;

procedure TgbCustomDataSet.InternalGotoBookmark(Bookmark: Pointer);
var
  x, Top: integer;
  OldCursor: TCursor;
  OldPosition: integer;
  Str: string;

  function GetWhere:string;
  var
    y, pos: integer;
    FieldStr: string;
  begin
    Result := ' ';
    for y := 0 to FKeyFields.Count - 1 do
    begin
      Str := CheckQuotes(Self, FKeyFields[y]);
      if FKeyFieldsForBookmark <> '' then
      begin
        pos := y + 1;
        FieldStr := ExtractFieldName(FKeyFieldsForBookmark, pos);
      end
      else
        FieldStr := Str;
      if y < (FKeyFields.Count - 1) then
        Result := Result + FieldStr + '=:' + Str + ' and '
      else
        Result := Result + FieldStr + '=:' + Str + ' ';
    end;
  end;

  function InWorkArea: Boolean;
  begin
    Result := (PInteger(Bookmark)^ >= Top) and (PInteger(Bookmark)^ <= (FRecordCount -1));
  end;

begin
  if DescNow then Top := FAboveDesc else Top := 0;
  if InWorkArea and CheckBmkRelative(Bookmark, PInteger(Bookmark)^) then
    FCurrentRecord := PInteger(Bookmark)^
  else
  begin
    for x := FRecordCount -1 downto Top do
      if CheckBmkRelative(Bookmark, x) then
      begin
        FCurrentRecord := x;
        Exit;
      end;
    OldPosition := PRecInfo(ActiveBuffer + FRecordSize).Bookmark;
    OldCursor := Screen.Cursor;
    Screen.Cursor := FSQLWaitCursor;
    try
     if FQLocate.Open then FQLocate.Close;
     FQLocate.SQL.Assign(FQSelect.SQL);
     InsertWhere(GetWhere, FQLocate.SQL);

     if opUseFirstStatement in FNavOptions then
     begin
       x := 0;
       Top := 0;
       while (x < 1) and (Top <= FQLocate.SQL.Count - 1) do
       begin
         x := Pos('SELECT', AnsiUpperCase(FQLocate.SQL[Top]));
         if x > 0 then
         begin
           Str := FQLocate.SQL[Top];
           System.Insert(' FIRST 1 ', Str, x + 6);
           FQLocate.SQL[Top] := Str;
           break;
         end;
         Inc(Top);
       end;
     end;

     SetParamsTo(FQLocate, nil, PChar(Bookmark), False);
     ExecWithCursor(FQLocate);
     if not RefreshAround(FQLocate, True) then
       FCurrentRecord := OldPosition;
    finally
     Screen.Cursor := OldCursor;
    end;
  end;
end;

{$IFDEF VER_FIBPLUS}
type
  THackDataBase = class(TFIBDataBase);
{$ENDIF}


procedure TgbCustomDataSet.InternalInitFieldDefs;
var
  FieldType: TFieldType;
  FieldSize: Word;
  IntTransaction : Boolean;
  i, FieldPosition, FieldPrecision, BmkOffs, FHole: Integer;
  FieldAliasName: string;
  RelationName, FieldName: string;

  function IsInKeyFields(const S: string): Boolean;
  var
    x, p: integer;
  begin
    Result := False;
    for x := 0 to FKeyFields.Count - 1 do
    begin
      p := Pos('.', FKeyFields[x]);
      if p > 0 then
      begin
        if (ExtractIdent(Self, Copy(FKeyFields[x], 1, p - 1)) = RelationName) and
          (ExtractIdent(Self, Copy(FKeyFields[x], p + 1, Length(S))) = FieldName) then
        begin
          Result := True;
          Exit;
        end;
      end
      else
      begin
        if S = ExtractIdent(Self, FKeyFields[x]) then
        begin
          Result := True;
          Exit;
        end;
      end;
    end;
  end;


begin
  if not Assigned(DataBase) then raise Exception.Create(gb_SNoDatabase);
  if not FQSelect.Prepared then
  begin
    IntTransaction := False;
    if not Assigned(FQSelect.DataBase) then
        FQSelect.DataBase := DataBase;
    if (not Assigned(FQSelect.Transaction)) or (not FQSelect.Transaction.Active) then
    begin
        IntTransaction := True;
{$IFDEF VER_IBX}
        FQSelect.Transaction := DataBase.InternalTransaction;
{$ELSE}
        FQSelect.Transaction := THackDataBase(DataBase).vInternalTransaction;
{$ENDIF}
        FQSelect.Transaction.StartTransaction;
    end;
    try
     FQSelect.Prepare;
    finally
     if IntTransaction then
     begin
       FQSelect.Transaction.Commit;
       FQSelect.Transaction := Transaction;
     end;
    end;
  end;

  ClearMapOfRecord;
  BmkOffs := SizeOf(Integer);
  BookmarkSize := SizeOf(Integer);

  FRecordSize := 0;
  FieldPosition := 0;
  FHole := 0;
  try
    FieldDefs.BeginUpdate;
    FieldDefs.Clear;
    for i := 0 to FQSelect.Current.Count - 1 do
{$IFNDEF VER_IBX}
      with FQSelect.Current[i].Data^ do
{$ELSE}
      with FQSelect.Current[i].Data do
{$ENDIF}
      begin
        AddToMapOfRecord;
        GetMap(i, False)^.FOffset := FRecordSize;
        Inc(FRecordSize);
        { Get the field name }
{$IFNDEF VER_IBX}
        SetString(FieldAliasName, aliasname, aliasname_length);
        SetString(RelationName, relname, relname_length);
        SetString(FieldName, sqlname, sqlname_length);
{$ELSE}
        FieldAliasName := aliasname;
        RelationName := relname;
        FieldName := sqlname;
{$ENDIF}
        FieldSize := 0;
        FieldPrecision := 0;

        GetMap(i, False)^.FType := sqltype and not 1;
        case sqltype and not 1 of
          { All VARCHAR's must be converted to strings before recording
           their values }
          SQL_VARYING, SQL_TEXT:
          begin
            FieldSize := sqllen;
            FieldType := ftString;
            FRecordSize := FRecordSize + sqllen + 1;
          end;
          { All Doubles/Floats should be cast to doubles }
          SQL_DOUBLE, SQL_FLOAT:
          begin
            FieldType := ftFloat;
            FRecordSize := FRecordSize + SizeOf(Double);
          end;
          SQL_SHORT:
          begin
            GetMap(i, False)^.FScale := sqlscale;
            if (sqlscale = 0) then
            begin
              FieldType := ftSmallInt;
              FRecordSize := FRecordSize + SizeOf(SmallInt);
            end
            else
            begin
              FieldType := ftBCD;
              FieldPrecision := 4;
              FieldSize := -sqlscale;
              FRecordSize := FRecordSize + SizeOf(Currency);
            end;
          end;
          SQL_LONG:
          begin
            GetMap(i, False)^.FScale := sqlscale;
            if (sqlscale = 0) then
            begin
              FieldType := ftInteger;
              FRecordSize := FRecordSize + SizeOf(Integer);
            end
            else
              if (sqlscale >= (-4)) then
              begin
                FieldType := ftBCD;
                FieldPrecision := 9;
                FieldSize := -sqlscale;
                FRecordSize := FRecordSize + SizeOf(Currency);
              end
              else
              begin
                FieldType := ftFloat;
                FRecordSize := FRecordSize + SizeOf(Double);
              end;
          end;
          SQL_INT64:
          begin
            GetMap(i, False)^.FScale := sqlscale;
            if (sqlscale = 0) then
            begin
              FieldType := ftLargeInt;
              FRecordSize := FRecordSize +  SizeOf(Int64);
            end
            else
              if (sqlscale >= (-4)) then
              begin
                FieldType := ftBCD;
                FieldPrecision := 18;
                FieldSize := -sqlscale;
                FRecordSize := FRecordSize + SizeOf(Currency);
              end
              else
              begin
                FieldType := ftFloat;
                FRecordSize := FRecordSize + SizeOf(Double);
              end;
          end;
          SQL_TIMESTAMP:
          begin
            FieldType := ftDateTime;
            FRecordSize := FRecordSize + SizeOf(TDateTime);
          end;
          SQL_TYPE_TIME:
          begin
            FieldType := ftTime;
            FRecordSize := FRecordSize + (SizeOf(TDateTime) div 2);
          end;
          SQL_TYPE_DATE:
          begin
            FieldType := ftDate;
            FRecordSize := FRecordSize + (SizeOf(TDateTime) div 2);
          end;
          SQL_BLOB:
          begin
            FieldSize := sizeof (TISC_QUAD);
            if (sqlsubtype = 1) then
              FieldType := ftmemo
            else
              FieldType := ftBlob;
            FRecordSize := FRecordSize + FieldSize;
          end;
          SQL_BOOLEAN:
          begin
            FieldType := ftBoolean;
            FRecordSize := FRecordSize + 4;
          end;
          SQL_ARRAY:
          begin
            FieldType := ftUnknown;
            Dec(FRecordSize);
          end;
          else
          begin
            FieldType := ftUnknown;
            Dec(FRecordSize);
          end;
        end;

        if (FieldType <> ftUnknown) then
        begin
          Inc(FieldPosition);
          with FieldDefs.AddFieldDef do
          begin
            Name := FieldAliasName;
            GetMap(i, False)^.FFieldName := Name;
            GetMap(i, False)^.FRelationNameForOrder := RelationName + '.' + FieldName;
            GetMap(i, False)^.FDataSize := (FRecordSize - GetMap(i, False)^.FOffset) - 1;
            GetMap(i, False)^.FUnknownType := False;
            GetMap(i, False)^.FNotUpdateble := False;
            if IsInKeyFields(Name) then
            begin
              GetMap(i, False)^.FBookmarkOffset := BmkOffs;
              BmkOffs := BmkOffs + (FRecordSize - GetMap(i, False)^.FOffset);
              BookmarkSize := BmkOffs;
            end
            else
              GetMap(i, False)^.FBookmarkOffset := 0;
            FieldNo := FieldPosition;
            DataType := FieldType;
            Size := FieldSize;
            Precision := FieldPrecision;
            Required := False;
            InternalCalcField := False;
{$IFDEF VER_IBX}
{$IFDEF D6}
            if (FieldName <> '') and (RelationName <> '') then
            begin
              if Database.Has_COMPUTED_BLR(RelationName, FieldName) then
              begin
                Attributes := [faReadOnly];
                InternalCalcField := True;
                GetMap(i, False)^.FNotUpdateble := True;
              end
            end;
{$ENDIF}
{$ENDIF}
            if ((SQLType and not 1) = SQL_TEXT) then
              Attributes := Attributes + [faFixed];
          end;
        end
        else
        begin
          GetMap(i, False)^.FUnknownType := True;
          Inc(FHole);
        end;
        GetMap(i, False)^.FHoleNumber := FHole;
      end;

    for i := 0 to FKeyFields.Count - 1 do
      FKeyFields[i] := ExtractIdent(Self, CheckRelation(FKeyFields[i], True));
  finally
    FieldDefs.EndUpdate;
  end;
end;

procedure TgbCustomDataSet.InternalInitRecord(Buffer: PChar);
begin
  FillChar(Buffer^, FRecordBufferSize, 0);
end;

procedure TgbCustomDataSet.InternalInsert;
begin
  CursorPosChanged;
end;

procedure TgbCustomDataSet.InternalLast;
var
  Buf: PChar;
begin
  if not FLastIsReal then
  begin
    if FQSelectDesc.Open then FQSelectDesc.Close;
    SetParamsTo(FQSelectDesc, nil, nil, False);
    ExecWithCursor(FQSelectDesc);
    FLastIsReal := True;
    FFirstIsReal := False;
    CurrentSelect := FQSelectDesc;
    FRecordCount := FBufferChunks;
    FCurrentRecord := FRecordCount - 1;
    if CurrentSelect.Next <> nil then
    begin
      Buf := AllocRecordBuffer;
      FetchRecordToBuffer(CurrentSelect, FCurrentRecord, Buf);
      FAboveDesc := FCurrentRecord;
      FreeRecordBuffer(Buf);
      Inc(FCurrentRecord);
    end
    else
      First;
  end
  else
    FCurrentRecord := FRecordCount;
end;

function TgbCustomDataSet.InternalLocateExt(const KeyFields: string; const KeyValues: Variant;
  Options: TLocateOptionsExt; IntOptions: TInternalLocateOptions): Boolean;
var
  OldCursor: TCursor;
begin
  CheckBrowseMode;
  if (KeyFields = '') or VarIsEmpty(KeyValues) then
  begin
    Result := False;
    Exit;
  end;
  OldCursor := Screen.Cursor;
  Screen.Cursor := FSQLWaitCursor;
  try
    if FQLocate.Open then FQLocate.Close;
    FQLocate.SQL.Assign(FQSelect.SQL);
    SetLocateWhere(KeyFields, KeyValues, Options, IntOptions, FQLocate);
    ExecWithCursor(FQLocate);
    Result := RefreshAround(FQLocate, True);
    if Result then Resync([rmCenter]);
  finally
    Screen.Cursor := OldCursor;
  end;
end;

procedure TgbCustomDataSet.InternalOpen;

  procedure GetKeyFields;
  var
    x: integer;
  begin
    FKeyFields.Clear;
    if FKeyFieldsForBookmark <> '' then
    begin
      x := 1;
      while x <= Length(FKeyFieldsForBookmark) do
        FKeyFields.Add(ExtractFieldName(FKeyFieldsForBookmark, x));
    end
    else
      GetFieldsInOrder_(FQSelect.SQL, FKeyFields);
  end;

  procedure ActivateSQL(IBSQL: {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF});
  begin
    if IBSQL.Open then IBSQL.Close;
    IBSQL.Database := FBase.Database;
    if (IBSQL <> FTecknics) then
      IBSQL.Transaction := FBase.Transaction;
    if (IBSQL <> FQSelect) and (IBSQL <> FQRefresh) and (IBSQL <> FQLocate)  then
      IBSQL.SQL.Assign(FQSelect.SQL);
  end;

  function CheckRefreshCorrect: Boolean;
  var
   x: integer;
  begin
    Result := False;
    if FQSelect.Current.Count <> FQRefresh.Current.Count then Exit;
    for x := 0 to FQSelect.Current.Count - 1 do
    begin
      if (FQSelect.Current[x].SQLType <> FQRefresh.Current[x].SQLType) or
        (FQSelect.Current[x].Size <> FQRefresh.Current[x].Size) or
        (FQSelect.Current[x].Name <> FQRefresh.Current[x].Name)
         then Exit;
    end;
    Result := True;
  end;

begin
  ActivateConnection;
{$IFDEF VER_FIBPLUS}
  FGDSLibrary := DataBase.ClientLibrary;
{$ENDIF}
  ActivateTransaction;
  ActivateSQL(FQSelect);
  SetFirstParams;
  ExecWithCursor(FQSelect);
  ActivateSQL(FQSelectDesc);
  ActivateSQL(FQSelectPart);
  ActivateSQL(FQSelectDescPart);
  ActivateSQL(FQRefresh);
  ActivateSQL(FTecknics);
  ActivateSQL(FQLocate);
  if opCommitOnPost in FUpdateOptions then
  begin
    FPostTransaction.DefaultDatabase := FBase.Database;
    FTecknics.Transaction := FPostTransaction;
  end
  else
    FTecknics.Transaction := FBase.Transaction;
  GetKeyFields;
  InternalInitFieldDefs;
  ConvertSQLs(False);
  if DefaultFields then
    CreateFields;
  BindFields(True);
  FOpen := FQSelect.Open;
  FCalcFieldsOffset := FRecordSize + SizeOf(TRecInfo);
  FRecordBufferSize := FCalcFieldsOffset + CalcFieldsSize;
  FCurrentRecord := -1;
  FRecordCount := 0;
  CheckCacheState(False);
  CurrentSelect := FQSelect;
  FQSelectDesc.Prepare;
  FQSelectPart.Prepare;
  FQSelectDescPart.Prepare;
  if FRefreshText.Text <> '' then
  begin
    FQRefresh.SQL.Assign(FRefreshText);
    FQRefresh.Prepare;
    if not CheckRefreshCorrect then
      raise Exception.Create(gb_SNotCorrectRefreshSQL);
  end;
  FFirstIsReal := True;
  FLastIsReal := False;
end;

procedure TgbCustomDataSet.InternalPost;
var
  Sucsess, EditFlag: Boolean;
begin
{$IFDEF D6}
  inherited;
{$ENDIF}
  case State of
    dsInsert, dsEdit:
    begin
      EditFlag := False;

      if (opCommitOnPost in FUpdateOptions) and (not FPostTransaction.Active) then
        FPostTransaction.StartTransaction;

      FinalizeWriteBlobs;

      if State = dsInsert then
      begin
        if opWhereChangeOnInsert in FUpdateOptions then
          FTecknics.SQL.Text := CheckForNulls(FInsertText, opWhereChangeOnInsert, False)
        else
          FTecknics.SQL.Assign(FInsertText);
      end
      else
      begin
        EditFlag := True;
        if opCheckNullsInWhere in FUpdateOptions then
           FTecknics.SQL.Text := CheckForNulls(FUpdateText, opCheckNullsInWhere, True)
        else
           FTecknics.SQL.Assign(FUpdateText);
      end;

      Sucsess := True;
      try
        if State = dsEdit then
          SetParamsTo(FTecknics, nil, nil, True)
        else
          SetParamsTo(FTecknics, nil, nil, False);
        ExecWithCursor(FTecknics);
      except
        if opCommitOnPost in FUpdateOptions then
          FPostTransaction.Rollback;
        raise;
      end;

      ClearWriteBlobs;

      try
        if FTecknics.RowsAffected = 0 then
        begin
          Sucsess := False;
          Cancel;
          raise EDatabaseError.Create(gb_SRecWasModified);
        end;
      finally
       if opCommitOnPost in FUpdateOptions then
       begin
         if Sucsess then FPostTransaction.Commit
         else
           if FPostTransaction.Active then FPostTransaction.Rollback;
       end;
       if (not Sucsess) and EditFlag then Next;
       if not Sucsess then
         Refresh
       else
         InternalRefresh;
      end;
    end;
  end;
end;

procedure TgbCustomDataSet.InternalRefresh;
var
  OldCursor: TCursor;
begin
  OldCursor := Screen.Cursor;
  Screen.Cursor := FSQLWaitCursor;
  try
   if (not EOF) or (([dsEdit, dsInsert] * [State]) <> [] ) then
   begin
     if FQRefresh.Open then FQRefresh.Close;
     if opCheckNullsInRefreshWhere in FNavOptions then
       FQRefresh.SQL.Text := CheckForNulls(FRefreshText, opCheckNullsInWhere, False);
     SetParamsTo(FQRefresh, nil, nil, False);
     ExecWithCursor(FQRefresh);
     while not RefreshAround(FQRefresh, False) do
     begin
       if (([dsEdit, dsInsert] * [State]) <> [] ) then
       begin
         FRefreshWithNextNeed := True;
         Exit;
       end;
       Next;
       if EOF then
       begin
         FLastIsReal := False;
         InternalLast;
         break;
       end;
       FQRefresh.Close;
       if opCheckNullsInRefreshWhere in FNavOptions then
         FQRefresh.SQL.Text := CheckForNulls(FRefreshText, opCheckNullsInWhere, False);
       SetParamsTo(FQRefresh, nil, nil, False);
       ExecWithCursor(FQRefresh);
     end;
   end
   else
   begin
     FLastIsReal := False;
     InternalLast;
   end;
  finally
    Screen.Cursor := OldCursor;
  end;
end;

procedure TgbCustomDataSet.InternalSetToRecord(Buffer: PChar);
begin
  FCurrentRecord := PRecInfo(Buffer + FRecordSize)^.Bookmark;
end;

function TgbCustomDataSet.IsCursorOpen: Boolean;
begin
  Result := FOpen;
end;

function TgbCustomDataSet.IsSequenced: Boolean;
begin
  Result := False;
end;

function TgbCustomDataSet.IsStatement(const S: string): Boolean;
begin
  Result := Pos('*' + AnsiUpperCase(S) + '*',
      '*ORDER*WHERE*SELECT*FROM*GROUP*HAVING*UNION*PLAN*') > 0;
end;

procedure TgbCustomDataSet.Loaded;
begin
  if Assigned(FBase.Database) and
{$IFDEF VER_IBX}
      (not FBase.Database.AllowStreamedConnected) and
{$ELSE}
      (not (ddoStoreConnected in FBase.Database.DesignDBOptions)) and
{$ENDIF}
      (not FBase.Database.Connected) and
       FStreamedActive then
    Active := False
  else
    if FStreamedActive then
      Active := True;
  inherited Loaded;
end;

function TgbCustomDataSet.LockCurrentRecord: Boolean;
begin
  if FTecknics.Open then FTecknics.Close;
  FTecknics.SQL.Text := FLockSQLText;
  try
    SetParamsTo(FTecknics, nil, nil, False);
    FTecknics.ExecQuery;
  except
    on E: {$IFDEF VER_IBX} EIBError {$ELSE} EFIBError {$ENDIF} do
    begin
      if FPostTransaction.Active then
        FPostTransaction.Rollback;
      if E.SQLCode = - 901 then
        raise EDatabaseError.Create(gb_SRecordIsLocked)
      else
        raise;
    end;
    else
    begin
      if FPostTransaction.Active then
        FPostTransaction.Rollback;
      raise;
    end;
  end;
  Result := FTecknics.RowsAffected > 0;
end;

function TgbCustomDataSet.Locate(const KeyFields: string;
  const KeyValues: Variant; Options: TLocateOptions): Boolean;

  function GetExtOption(Opt: TLocateOptions): TLocateOptionsExt;
  begin
    Result := [];
    if loCaseInsensitive in Opt then Include(Result, leCaseInsensitive);
    if loPartialKey in Opt then Include(Result, lePartialKey);
  end;

begin
  Result := InternalLocateExt(KeyFields, KeyValues, GetExtOption(Options),[]);
end;

function TgbCustomDataSet.LocateExt(const KeyFields: string;
  const KeyValues: Variant; Options: TLocateOptionsExt): Boolean;
begin
  Result := InternalLocateExt(KeyFields, KeyValues, Options, []);
end;

function TgbCustomDataSet.LocateNearest(const KeyFields: string;
  const KeyValues: Variant): Boolean;

 function CheckFieldsInOrder: Boolean;
 var
   x, p: integer;
   List: TStringList;
 begin
   Result := False;
   List := TStringList.Create;
   try
     GetFieldsInOrder_(FQSelect.SQL, List);
     x := 0;
     p := 1;
     while  (p <= Length(KeyFields)) and (x < List.Count) do
     begin
       if ExtractFieldName(KeyFields, p) <>
         ExtractIdent(Self, CheckRelation(List[x], False)) then Exit;
       Inc(x);
     end;
     Result := True;
   finally
     List.Free;
   end;
 end;

begin
  if CheckFieldsInOrder then
  begin
    Result := InternalLocateExt(KeyFields, KeyValues, [], [ioNearest]);
    if not Result then
    begin
      Last;
      Result := True;
    end;
  end
  else
    Result := InternalLocateExt(KeyFields, KeyValues, [], []);
end;

function TgbCustomDataSet.LocateNext(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptionsExt): Boolean;
begin
  Result := InternalLocateExt(KeyFields, KeyValues, Options, [ioNextFlag]);
end;

function TgbCustomDataSet.LocatePrior(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptionsExt): Boolean;
begin
  Result := InternalLocateExt(KeyFields, KeyValues, Options, [ioPriorFlag]);
end;

function TgbCustomDataSet.Lookup(const KeyFields: string;
  const KeyValues: Variant; const ResultFields: string): Variant;
var
  P, ACount: Integer;
  S: string;
  V: array of Variant;
begin
  if (KeyFields = '') or VarIsEmpty(KeyValues) then
  begin
    Result := Null;
    Exit;
  end;
  if FQLocate.Open then FQLocate.Close;
  FQLocate.SQL.Assign(FQSelect.SQL);
  SetLocateWhere(KeyFields, KeyValues, [], [], FQLocate);
  ExecWithCursor(FQLocate);
  if FQLocate.Next <> nil then
  begin
    P := 1;
    ACount := 0;
    while P <= Length(ResultFields) do
    begin
      S := ExtractFieldName(ResultFields, P);
      Inc(ACount);
      SetLength(V, ACount);
{$IFDEF VER_IBX}
      V[ACount - 1] := FQLocate.FieldByName(S).Value;
{$ELSE}
      V[ACount - 1] := FQLocate.Fields[FQLocate.FieldIndex[S]].Value;
{$ENDIF}
    end;
    if ACount = 1 then
      Result := V[0]
    else
      Result := VarArrayOf(V);
  end
  else
    Result := False;
end;

procedure TgbCustomDataSet.MoveInCache(CurIndex, NewIndex: Integer);
var
  Buffer: PChar;
  x: integer;
begin
  Buffer := FCache[CurIndex];
  if CurIndex < NewIndex then
    Move(FCache[CurIndex + 1], FCache[CurIndex],
      (NewIndex - CurIndex) * SizeOf(Pointer))
  else
    Move(FCache[NewIndex], FCache[NewIndex + 1],
      (CurIndex - NewIndex) * SizeOf(Pointer));
  FCache[NewIndex] := Buffer;
  for x := 0 to FRecordCount - 1 do
  begin
    Buffer := FCache[x];
    PRecInfo(Buffer + FRecordSize)^.Bookmark := x;
  end;
end;

procedure TgbCustomDataSet.OrderToDesc(SQL, SList: TStrings);
var
  CurPos, NextPos, x: integer;
  S, Buf: string;
begin
  S := SQL.Text;
  NextPos := 0;
  CurPos := GetPosOfStatement(S, 'order', NextPos);
  if CurPos > 0 then
  begin
    Buf := 'order by';
    for x:= 0 to SList.Count - 1 do
    begin
      Buf := Buf + ' ' + SList[x];
      if Integer(SList.Objects[x]) = 0 then
        Buf := Buf + ' ' + 'DESC';
      if x < SList.Count - 1 then
        Buf := Buf + ',';
    end;

    System.Delete(S, CurPos, NextPos - CurPos);
    System.Insert(Buf, S, CurPos);
    SQL.Text := S;
  end;
end;

procedure TgbCustomDataSet.ReadFromCache(ARecNo: integer; Buffer: PChar);
begin
  Move(FCache[ARecNo]^, Buffer^, FRecordBufferSize);
end;

function TgbCustomDataSet.RefreshAround(IBSQL: {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF}; Center: Boolean):Boolean;
var
  Buf : PChar;
  x, Delta, y: integer;
begin
  Result := False;
  if IBSQL.Next <> nil then
  begin
    Result := True;
    Buf := AllocRecordBuffer;
    FRecordCount := BufferCount + 1;
    if Center then
      FCurrentRecord := FRecordCount div 2
    else
      FCurrentRecord := ActiveRecord;
    FetchRecordToBuffer(IBSQL, FCurrentRecord, Buf);

    FFirstIsReal := False;
    FLastIsReal := False;

    if FQSelectDescPart.Open then FQSelectDescPart.Close;
    SetParamsTo(FQSelectDescPart, IBSQL, nil, False);
    ExecWithCursor(FQSelectDescPart);
    Delta := FCurrentRecord;
    for x := FCurrentRecord - 1 downto 0 do
    begin
      if FQSelectDescPart.Next <> nil then
      begin
        FetchRecordToBuffer(FQSelectDescPart, x, Buf);
        FAboveDesc := x;
      end
      else
      begin
        InternalFirst;
        for y := 0 to (Delta - 1) - x  do
        begin
          Inc(FCurrentRecord);
          CurrentSelect.Next;
          FetchRecordToBuffer(CurrentSelect, y, Buf);
          Inc(FRecordCount);
        end;
        FreeRecordBuffer(Buf);
        Exit;
      end;
    end;

    if FQSelectPart.Open then FQSelectPart.Close;
    SetParamsTo(FQSelectPart, IBSQL, nil, False);
    ExecWithCursor(FQSelectPart);
    Delta := 0;
    for x := FCurrentRecord + 1 to FRecordCount -1  do
    begin
      if FQSelectPart.Next <> nil then
      begin
        FetchRecordToBuffer(FQSelectPart, x, Buf);
        Inc(Delta);
      end
      else
      begin
        InternalLast;
        Dec(FCurrentRecord);
        for y := FCurrentRecord - 1 downto FCurrentRecord - Delta  do
        begin
          Dec(FCurrentRecord);
          CurrentSelect.Next;
          FetchRecordToBuffer(CurrentSelect, y, Buf);
          FAboveDesc := y;
        end;
        FreeRecordBuffer(Buf);
        Exit;
      end;
    end;

    CurrentSelect := FQSelectPart;
    FreeRecordBuffer(Buf);
  end;
end;

procedure TgbCustomDataSet.RefreshCurrentRecord(InPostTransaction: Boolean);
begin
  try
   if FQRefresh.Open then FQRefresh.Close;
   if InPostTransaction then
     FQRefresh.Transaction := FPostTransaction;
   if opCheckNullsInRefreshWhere in FNavOptions then
     FQRefresh.SQL.Text := CheckForNulls(FRefreshText, opCheckNullsInWhere, False);
   SetParamsTo(FQRefresh, nil, nil, False);
   FQRefresh.ExecQuery;
   if FQRefresh.Next <> nil then
   begin
     FetchRecordToBuffer(FQRefresh, FCurrentRecord, ActiveBuffer);
   end
   else
   begin
     if FPostTransaction.Active then FPostTransaction.Rollback;
     Refresh;
     raise EDatabaseError.Create(gb_SRecWasModified);
   end;
  finally
    if InPostTransaction then
      FQRefresh.Transaction := FBase.Transaction;
  end;
end;

procedure TgbCustomDataSet.ReorderCache(Direction: Boolean);
var
  Count, CountBytes, x: integer;
  Buf: PChar;
begin
  Count := (FBufferChunks div 2);
  CountBytes := Count * SizeOf(Pointer);
  if Direction then
  begin
    Move(FCache[Count], TempBuffer^, CountBytes);
    Move(FCache[0], FCache[Count], CountBytes);
    Move(TempBuffer^, FCache[0], CountBytes);
    FRecordCount := FRecordCount - Count;
    FCurrentRecord := FCurrentRecord - Count;
    for x := 0 to Count - 1 do
    begin
      Buf := FCache[x];
      PRecInfo(Buf + FRecordSize)^.Bookmark := PRecInfo(Buf + FRecordSize)^.Bookmark - Count;
    end;
    for x := 0 to BufferCount - 1 do
    begin
      Buf := Buffers[x];
      PRecInfo(Buf + FRecordSize)^.Bookmark := PRecInfo(Buf + FRecordSize)^.Bookmark - Count;
    end;
    FFirstIsReal := False;
  end
  else
  begin
    Move(FCache[0], TempBuffer^, CountBytes);
    Move(FCache[Count], FCache[0], CountBytes);
    Move(TempBuffer^, FCache[Count], CountBytes);
    FRecordCount := FRecordCount + Count;
    if FRecordCount >  BufferChunks then
    begin
      FRecordCount := BufferChunks;
    end;
    FCurrentRecord := FCurrentRecord + Count;
    if FCurrentRecord >= BufferChunks then
        FCurrentRecord := BufferChunks - 1;
    for x := Count to High(FCache) do
    begin
      Buf := FCache[x];
      PRecInfo(Buf + FRecordSize)^.Bookmark := PRecInfo(Buf + FRecordSize)^.Bookmark + Count;
    end;
    for x := 0 to BufferCount - 1 do
    begin
      Buf := Buffers[x];
      PRecInfo(Buf + FRecordSize)^.Bookmark := PRecInfo(Buf + FRecordSize)^.Bookmark + Count;
    end;
    FLastIsReal := False;
  end;
end;

procedure TgbCustomDataSet.ReQuery(KeepPosition: Boolean);
begin
  CheckBrowseMode;
  if KeepPosition and (not IsEmpty) then
  begin
    if EOF then
    begin
      DisableControls;
      try
        if MoveBy(-1) < 0 then MoveBy(1);
      finally
        EnableControls;
      end;
    end;
    if FQRefresh.Open then FQRefresh.Close;
    if opCheckNullsInRefreshWhere in FNavOptions then
      FQRefresh.SQL.Text := CheckForNulls(FRefreshText, opCheckNullsInWhere, False);
    SetParamsTo(FQRefresh, nil, nil, False);
    ExecWithCursor(FQRefresh);
    if not RefreshAround(FQRefresh, False) then
    begin
      FFirstIsReal := False;
      First;
    end
    else
      Resync([]);
  end
  else
  begin
    FFirstIsReal := False;
    First;
  end;
end;

procedure TgbCustomDataSet.SetActive(Value: Boolean);
begin
  if (csReading in ComponentState) and
     (not (csDesigning in ComponentState)) then
    FStreamedActive := true
  else
    inherited SetActive(Value);
end;

procedure TgbCustomDataSet.SetBookmarkData(Buffer: PChar; Data: Pointer);
begin
  PRecInfo(Buffer + FRecordSize).Bookmark := PInteger(Data)^;
end;

procedure TgbCustomDataSet.SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag);
begin
  PRecInfo(Buffer + FRecordSize).BookmarkFlag := Value;
end;

procedure TgbCustomDataSet.SetDatabase(Value: {$IFDEF VER_IBX} TIBDatabase {$ELSE} TFIBDatabase {$ENDIF});
begin
  if Value <> FBase.Database then FBase.DataBase := Value;
  if (not Assigned(FBase.Transaction)) and (FBase.Database <> nil)  then
    FBase.Transaction := FBase.Database.DefaultTransaction;
end;

procedure TgbCustomDataSet.SetDeleteSQL(Value: TStrings);
begin
  if FDeleteText.Text <> Value.Text then
    FDeleteText.Assign(Value);
end;

procedure TgbCustomDataSet.SetFieldData(Field: TField; Buffer: Pointer);
var
  PInfo: PFieldInfo;
  lTempCurr: System.Currency;
  CurBuf: PChar;
begin
  CurBuf := GetActiveBuf;
  if Field.FieldNo > 0 then
  begin
    if (([dsEdit, dsInsert, dsSetKey, dsCalcFields, dsFilter, dsNewValue,
       dsInternalCalc] * [State]) = []) then raise Exception.Create(gb_SNotInEditMode);
    PInfo := GetMap(Field.FieldNo - 1, True);
    if Buffer <> nil then
    begin
      if (Field.DataType = ftBCD) then
      begin
        BCDToCurr(TBCD(Buffer^), lTempCurr);
        Move(PChar(@lTempCurr)^, PChar(CurBuf + PInfo^.FOffset)^, PInfo^.FDataSize);
      end
      else
        Move(Buffer^, PChar(CurBuf + PInfo^.FOffset)^, PInfo^.FDataSize);
      Boolean(CurBuf[PInfo^.FOffset + PInfo^.FDataSize]) := Boolean(True);
    end
    else
      Boolean(CurBuf[PInfo^.FOffset + PInfo.FDataSize]) := Boolean(False);
    SetModified(True);
    if not (State in [dsCalcFields, dsFilter, dsNewValue]) then
       DataEvent(deFieldChange, Longint(Field));
  end
  else
  begin
    Inc(CurBuf, FCalcFieldsOffset + Field.Offset);
    Boolean(CurBuf[0]) := LongBool(Buffer);
    if Boolean(CurBuf[0]) then
    begin
      Move(Buffer^, CurBuf[1], Field.DataSize);
    end;
  end;
end;

procedure TgbCustomDataSet.SetFieldData(Field: TField; Buffer: Pointer;
  NativeFormat: Boolean);
var
  PInfo: PFieldInfo;
  CurBuf: PChar;
begin
  if (Field.DataType = ftBCD) and (not NativeFormat) and (Field.FieldNo > 0) then
  begin
    CurBuf := GetActiveBuf;
    PInfo := GetMap(Field.FieldNo - 1, True);
    if Buffer <> nil then
    begin
      Move(Buffer^, PChar(CurBuf + PInfo^.FOffset)^, PInfo^.FDataSize);
      Boolean(CurBuf[PInfo^.FOffset + PInfo^.FDataSize]) := Boolean(True);
    end
    else
      Boolean(CurBuf[PInfo^.FOffset + PInfo.FDataSize]) := Boolean(False);
    SetModified(True);
    if not (State in [dsCalcFields, dsFilter, dsNewValue]) then
      DataEvent(deFieldChange, Longint(Field));
  end
  else
    inherited SetFieldData(Field, Buffer, NativeFormat);
end;

procedure TgbCustomDataSet.SetFirstParams;
var
 x: integer;
begin
  for x := 0 to FQSelect.Params.Count - 1 do
    if Pos('M$S$T$_', FQSelect.Params[x].Name) = 1 then
      FQSelect.Params[x].Value := GetMasterFieldValue(Copy(FQSelect.Params[x].Name, 8, Length(FQSelect.Params[x].Name)))
    else
{$IFDEF VER_IBX}
      FQSelect.Params[x].Value := FSQLParams.ByName(FQSelect.Params[x].Name).Value;
{$ELSE}
      FQSelect.Params[x].Value := FSQLParams.ByName[FQSelect.Params[x].Name].Value;
{$ENDIF}
end;

procedure TgbCustomDataSet.SetInsertSQL(Value: TStrings);
begin
  if FInsertText.Text <> Value.Text then
    FInsertText.Assign(Value);
end;

procedure TgbCustomDataSet.SetLocateWhere(const KeyFields: string; const KeyValues: Variant;
   Options: TLocateOptionsExt; IntOptions: TInternalLocateOptions; IBSQL: {$IFDEF VER_IBX} TIBSQL {$ELSE} TFIBQuery {$ENDIF});
var
  fString, pString, eString, Str, Par2: String;
  x, Index: integer;
  val: Array of Variant;
  FieldList: TStringList;

  procedure ExtrNames;
  var
     P: integer;
  begin
    P := 1;
    while P <= Length(KeyFields) do
      FieldList.Add(ExtractFieldName(KeyFields, P));
  end;

begin
  FieldList := TStringList.Create;
  try
   ExtrNames;
   SetLength(val, FieldList.Count);
   for x := 0 to FieldList.Count - 1 do
   if VarIsArray(KeyValues) then
     val[x] := KeyValues[x]
   else
     val[x] := KeyValues;
   Str := '';

   for x := 0 to FieldList.Count - 1 do
   begin
     if (ioNearest in IntOptions) and (x = FieldList.Count - 1)  then
       eString := ' >= '  else  eString :=  ' = ';

{$IFDEF D6}
     if VarIsStr(val[x]) then
{$ELSE}
     if VarType(val[x]) = varString then
{$ENDIF}
     begin
       if (leCaseInsensitive in Options) then
       begin
         if opUseCyrCollate in FNavOptions then
           fString := 'UPPER(%:0s ' + CyrCollate + ')'
         else
           fString := 'UPPER(%:0s)';
       end
       else
         fString := '%:0s';

       if lePartialKey in Options then eString := ' starting with ';
       if leContaining in Options then eString := ' containing ';
     end
     else
       fString := '%:0s';

     Par2 := CheckRelation(FieldList[x], False);
     pString := ':' + CheckQuotes(Self, 'L$C$T$_' + Par2) + ' ';

     if x > 0 then
     begin
       if VarIsNull(val[x]) then
         Str := Str + Format(' and %s is null ', [CheckQuotes(Self, FieldList[x])])
       else
         Str := Str + Format(' and ' + fString + eString + pString, [CheckQuotes(Self, FieldList[x])]);
     end
     else
     begin
       if VarIsNull(val[x]) then
         Str := Format(' %s is null ', [CheckQuotes(Self, FieldList[x])])
       else
         Str := Format(' ' + fString + eString + pString, [CheckQuotes(Self, FieldList[x])]);
     end;
     if Par2 <> FieldList[x] then
       FieldList[x] := Par2;
   end;

   InsertWhere(Str, IBSQL.SQL);

   if opUseFirstStatement in FNavOptions then
   begin
     x := 0;
     Index := 0;
     while (x < 1) and (Index <= (IBSQL.SQL.Count - 1)) do
     begin
       x := Pos('SELECT', AnsiUpperCase(IBSQL.SQL[Index]));
       if x > 0 then
       begin
         Str := IBSQL.SQL[Index];
         System.Insert(' FIRST 1 ', Str, x + 6);
         IBSQL.SQL[Index] := Str;
         break;
       end;
       Inc(Index);
     end;
   end;

   if (ioNextFlag in IntOptions) or (ioPriorFlag in IntOptions) then
   begin
     SetNextPriorWhere(IBSQL.SQL, (ioNextFlag in IntOptions));
     SetParamsTo(IBSQL, nil, nil, False);
   end;

   for x := 0 to IBSQL.Params.Count -1 do
   begin
     if (Pos('M$S$T$_', IBSQL.Params[x].Name) = 1) and (not ((ioNextFlag in IntOptions) or (ioPriorFlag in IntOptions))) then
       IBSQL.Params[x].Value := GetMasterFieldValue(Copy(IBSQL.Params[x].Name, 8, Length(IBSQL.Params[x].Name)))
     else
     if Pos('L$C$T$_', IBSQL.Params[x].Name) = 1 then
     begin
       pString := Copy(IBSQL.Params[x].Name, 8, Length(IBSQL.Params[x].Name));
       Index := FieldList.IndexOf(pString);
       if (leCaseInsensitive in Options) and {$IFDEF D6} VarIsStr(val[Index]) {$ELSE}
           (VarType(val[Index]) = varString)  {$ENDIF}  then
         IBSQL.Params[x].Value := AnsiUpperCase(val[Index])
       else
       begin
{$IFDEF D6}
         if VarIsType(val[Index], varInt64) then
           IBSQL.Params[x].AsInt64 := val[Index]
         else
{$ELSE}
         if VarType(val[Index]) = 14 then
            IBSQL.Params[x].AsInt64 := Round(val[Index])
         else
{$ENDIF}
           IBSQL.Params[x].Value := val[Index];
       end;
     end
     else
     if (not ((ioNextFlag in IntOptions) or (ioPriorFlag in IntOptions))) then
{$IFDEF VER_IBX}
       IBSQL.Params[x].Value := FSQLParams.ByName(IBSQL.Params[x].Name).Value;
{$ELSE}
       IBSQL.Params[x].Value := FSQLParams.ByName[IBSQL.Params[x].Name].Value;
{$ENDIF}
   end;
  finally
    FieldList.Free;
    val := nil;
  end;
end;

procedure TgbCustomDataSet.SetNextPriorWhere(SQL: TStrings; ANext: Boolean);
var
  SList: TStringList;
begin
  SList := TStringList.Create;
  GetFieldsInOrder_(SQL, SList);
  InsertWhere(GenerateWhere(SList, ANext), SQL);
  if not ANext then OrderToDesc(SQL, SList);
  SList.Free;
end;

procedure TgbCustomDataSet.SetParamsTo(IBSQL, From: {$IFDEF VER_IBX} TIBSQL
        {$ELSE} TFIBQuery {$ENDIF}; FromBuffer: PChar; OldBufferNeed: Boolean);
var
  x, LocalInt: integer;
  CurBuffer ,Buffer, OldBuffer, FieldBuf : PChar;
  PInfo: PFieldInfo;
  Par: string;
  ts: TTimeStamp;
begin
  Buffer := nil;
  if From <> nil then
  begin
    Buffer := AllocRecordBuffer;
    FetchRecordToBuffer(From, -1, Buffer);
  end
  else
  begin
    if FromBuffer <> nil then
      Buffer := FromBuffer
    else
      Buffer := ActiveBuffer;
    if not Assigned(Buffer) then Exit;
  end;

  OldBuffer := nil;
  if not OldBufferNeed then OldBuffer := Buffer;

  for x := 0 to IBSQL.Params.Count - 1 do
  begin
    Par := ExtractIdent(Self, IBSQL.Params[x].Name);
    if Pos('OLD_', Par) = 1 then
    begin
      if OldBuffer = nil then
      begin
        OldBuffer := AllocRecordBuffer;
        ReadFromCache(FCurrentRecord, OldBuffer);
      end;
      Par := Copy(Par, 5, Length(Par));
      CurBuffer := OldBuffer;
    end
    else
      CurBuffer := Buffer;

    if Pos('M$S$T$_', Par) = 1 then
    begin
      IBSQL.Params[x].Value := GetMasterFieldValue(Copy(Par, 8, Length(Par)));
      Continue;
    end;

    PInfo := GetMapByName(Par);

    if PInfo = nil then
    begin
      if Pos('L$C$T$_', Par) = 0 then
{$IFDEF VER_IBX}
        IBSQL.Params[x].Value := FSQLParams.ByName(Par).Value;
{$ELSE}
        IBSQL.Params[x].Value := FSQLParams.ByName[Par].Value;
{$ENDIF}
      Continue;
    end;

    if FromBuffer = nil then
      FieldBuf := CurBuffer + PInfo^.FOffset
    else
      FieldBuf := CurBuffer + PInfo^.FBookmarkOffset;

    if not Boolean(FieldBuf[PInfo^.FDataSize]) then
    begin
      IBSQL.Params[x].IsNull := True;
    end
    else
    begin
      IBSQL.Params[x].IsNull := False;
      case PInfo^.FType of
        SQL_TEXT, SQL_VARYING:
        begin
          IBSQL.Params[x].AsString := StrPas(FieldBuf);
        end;
        SQL_FLOAT, SQL_DOUBLE, SQL_D_FLOAT:
          IBSQL.Params[x].AsDouble := PDouble(FieldBuf)^;
        SQL_SHORT, SQL_LONG:
        begin
          if PInfo^.FScale = 0 then
          begin
            if PInfo^.FType = SQL_LONG then
              IBSQL.Params[x].AsLong := PLong(FieldBuf)^
            else
            begin
              LocalInt := PShort(FieldBuf)^;
              IBSQL.Params[x].AsLong := LocalInt;
            end;
          end
          else
          if PInfo^.FScale >= (-4) then
            IBSQL.Params[x].AsCurrency := PCurrency(FieldBuf)^
          else
            IBSQL.Params[x].AsDouble := PDouble(FieldBuf)^;
        end;
        SQL_INT64:
        begin
          if PInfo^.FScale = 0 then
            IBSQL.Params[x].AsInt64 := PInt64(FieldBuf)^
          else
          if PInfo^.FScale >= (-4) then
            IBSQL.Params[x].AsCurrency := PCurrency(FieldBuf)^
          else
            IBSQL.Params[x].AsDouble := PDouble(FieldBuf)^;
        end;
        SQL_BLOB, SQL_ARRAY, SQL_QUAD:
          IBSQL.Params[x].AsQuad := PISC_QUAD(FieldBuf)^;
        SQL_TYPE_DATE:
        begin
          ts.Date := PInt(FieldBuf)^;
          ts.Time := 0;
          IBSQL.Params[x].AsDate := TimeStampToDateTime(ts);
        end;
        SQL_TYPE_TIME:
        begin
          ts.Date := 1;
          ts.Time := PInt(FieldBuf)^;
          IBSQL.Params[x].AsTime := TimeStampToDateTime(ts);
        end;
        SQL_TIMESTAMP:
          IBSQL.Params[x].AsDateTime := TimeStampToDateTime(MSecsToTimeStamp(PDouble(FieldBuf)^));
        SQL_BOOLEAN:
          IBSQL.Params[x].AsBoolean := (PShort(FieldBuf)^ = ISC_TRUE);
      end;
    end;
  end;

  if Buffer <> nil then
    if (Buffer <> ActiveBuffer) and (Buffer <> FromBuffer) then
      FreeRecordBuffer(Buffer);
  if (OldBuffer <> nil) and OldBufferNeed then
    FreeRecordBuffer(OldBuffer);
end;

procedure TgbCustomDataSet.SetRefreshSQL(Value: TStrings);
begin
  if FRefreshText.Text <> Value.Text then
    FRefreshText.Assign(Value);
  if FQRefresh.Open then FQRefresh.Close;
  FQRefresh.SQL.Assign(Value);
end;

procedure TgbCustomDataSet.SetSelectSQL(Value: TStrings);
begin
  if FQSelect.SQL.Text <> Value.Text then
  begin
    if Active then Close;
    FQSelect.SQL.Assign(Value);
  end;
end;

procedure TgbCustomDataSet.SetTransaction(const Value: {$IFDEF VER_IBX} TIBTransaction {$ELSE} TFIBTransaction {$ENDIF});
begin
  if Value <> FBase.Transaction then FBase.Transaction := Value;
end;

procedure TgbCustomDataSet.SetUpdateSQL(Value: TStrings);
begin
  if FUpdateText.Text <> Value.Text then
      FUpdateText.Assign(Value);
end;

function TgbCustomDataSet.TokenNext(const S: string; BlankInt: Boolean; var CurPos, CurEnd: integer):string;
var
  Blank: set of char;
begin
  Result := '';
  CurEnd := CurPos;
  while (CurPos <= Length(S)) and (S[CurPos]<=' ') do Inc(CurPos);
  if CurPos > Length(S) then Exit;
  if S[CurPos] in IdentChars then
  begin
    CurEnd := CurPos;
    Blank := [];
    while (CurEnd <= Length(S))
      and (S[CurEnd] in IdentChars + Blank) do
    begin
      if BlankInt and (S[CurEnd] = '"') then
      begin
        if (' ' in Blank) then Exclude(Blank, ' ')
                          else Include(Blank, ' ');
      end;
      Inc(CurEnd);
    end;
    Result := Copy(S, CurPos, (CurEnd - CurPos));
  end
  else
  begin
    Result := S[CurPos];
    CurEnd := CurPos + 1;
  end;
end;

procedure TgbCustomDataSet.WriteToCache(ARecNo: integer; Buffer: PChar);
begin
  Move(Buffer^, FCache[ARecNo]^, FRecordBufferSize);
end;

procedure TgbCustomDataSet.InternalHandleException;
begin
  //
end;

function TgbCustomDataSet.CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream;
var
  Buff: PChar;
  PInfo: PFieldInfo;
  Bs: TgbIBBlobStream;
  TempQuad: TISC_QUAD;

  function IsInList: Boolean;
  var
    x: integer;
  begin
    Result := False;
    for x := 0 to FWriteBlobs.Count - 1 do
    begin
      if Field = TgbIBBlobStream(FWriteBlobs[x]).FField then
      begin
        Result := True;
        Bs := TgbIBBlobStream(FWriteBlobs[x]);
        break;
      end;
    end;
  end;

begin
  if not IsInList then
  begin
    Buff := GetActiveBuf;
    Bs := TgbIBBlobStream.Create;
    if Mode in  [bmWrite, bmReadWrite] then
      Bs.Mode := bmReadWrite
    else
      Bs.Mode := Mode;
    if Buff <> nil then
    begin
      PInfo := GetMap(Field.FieldNo - 1, True);
      if Boolean(PChar(Buff + PInfo^.FOffset)[PInfo^.FDataSize]) then
        Bs.BlobID := PISC_QUAD(PChar(Buff + PInfo^.FOffset))^
      else
      begin
        TempQuad.gds_quad_high := 0;
        TempQuad.gds_quad_low := 0;
        BS.BlobID := TempQuad;
      end;
    end;
    Bs.Database := Database;
    if (Mode = bmWrite) and (opCommitOnPost in FUpdateOptions) then
    begin
      if not FPostTransaction.Active then
        FPostTransaction.StartTransaction;
      Bs.Transaction := FPostTransaction;
    end
    else
      Bs.Transaction := Transaction;
    Bs.FField := Field;
    if Mode = bmWrite then
      FWriteBlobs.Add(Bs);
  end;
  Result := TgbBlobStream.Create(Field, Bs, Mode);
end;

procedure TgbCustomDataSet.ClearWriteBlobs;
var
  x: integer;
begin
  if FWriteBlobs.Count = 0 then Exit;
  for x := 0 to FWriteBlobs.Count - 1 do
    TgbIBBlobStream(FWriteBlobs[x]).Free;
  FWriteBlobs.Clear;
end;

procedure TgbCustomDataSet.FinalizeWriteBlobs;
var
  PInfo: PFieldInfo;
  x: integer;
begin
  for x := 0 to FWriteBlobs.Count - 1 do
  begin
    TgbIBBlobStream(FWriteBlobs[x]).Finalize;
    PInfo := GetMap(TgbIBBlobStream(FWriteBlobs[x]).FField.FieldNo - 1, True);
    PISC_QUAD(PChar(ActiveBuffer + PInfo^.FOffset))^ := TgbIBBlobStream(FWriteBlobs[x]).BlobID;
    if TgbIBBlobStream(FWriteBlobs[x]).Size = 0 then
      Boolean(ActiveBuffer[PInfo^.FOffset + PInfo^.FDataSize]) := Boolean(False)
    else
      Boolean(ActiveBuffer[PInfo^.FOffset + PInfo^.FDataSize]) := Boolean(True);
  end;
end;

{ TGeneratorDefs }

function TGeneratorDefs.GetGeneratorDef(Index: Integer): TGeneratorDef;
begin
  Result := TGeneratorDef(inherited Items[Index]);
end;

procedure TGeneratorDefs.SetGeneratorDef(Index: Integer;
  Value: TGeneratorDef);
begin
  inherited Items[Index] := Value;
end;

{ TGeneratorDef }

constructor TGeneratorDef.Create(Collection: TCollection);
begin
  inherited;
  FIncrementBy := 1;
  FGeneratorType := gtOnInsert;
end;

function TGeneratorDef.GetDisplayName: string;
begin
  Result := 'Gen-or: ' + FGeneratorName + ' on field: ' + FFieldName;
end;

{ TgbDataSet }

{$IFDEF VER_FIBPLUS}
type
  THackFIBXSQLDA = class(TFIBXSQLDA);
{$ENDIF}

constructor TgbDataSet.Create(AOwner: TComponent);
begin
  inherited;
  FQSelect.OnSQLChanging := SQLChanging;
  if FUseMacros then
  begin
    FInsertText.OnChanging := SQLChanging;
    FUpdateText.OnChanging := SQLChanging;
    FDeleteText.OnChanging := SQLChanging;
    FRefreshText.OnChanging := SQLChanging;
  end;
  FQSelectAlreadyInArray := False;
  FSQLSaved := False;
{$IFDEF VER_IBX}
  FSQLParams := TIBXSQLDA.Create(FQSelect);
{$ELSE}
  FSQLParams := TFIBXSQLDA.Create(True);
  THackFIBXSQLDA(FSQLParams).FQuery := FQSelect;
{$ENDIF}
  FParamsAssigned := False;
  FMacroChar := '%';
  FUseMacros := False;
  FMacrosDef := TMacrosDef.Create(Self);
  FCurrentSQLInSynchr := -1;
  FSynchrList := TStringList.Create;
  FSynchrList.OnChange := SynchrChanged;
end;

destructor TgbDataSet.Destroy;
begin
  inherited;
  FSQLParams.Free;
  FMacrosDef.Free;
  FSynchrList.Free;
end;

procedure TgbDataSet.ActivateFQSelect;
begin
  FQSelect.Database := FBase.Database;
  FQSelect.Transaction := FBase.Transaction;
  ActivateConnection;
  ActivateTransaction;
  if not FUseMacros then
    FQSelect.Prepare
  else
  begin
    try
      Include(IdentChars, FMacroChar);
      FSavedSQLs[0] := FQSelect.SQL.Text;
      FQSelect.SQL.Text := ExpandSQL(FSavedSQLs[0]);
      FQSelect.Prepare;
      FQSelectAlreadyInArray := True;
    except
      FQSelect.SQL.Text := FSavedSQLs[0];
      FQSelectAlreadyInArray := False;
    end;
  end;
end;

procedure TgbDataSet.AssignParams;
var
 x: integer;
 TempList: TList;

  function CheckCorrectParamName(ParName: string): Boolean;
  var
   i: integer;
  begin
    Result := True;
    for i := 0 to FQSelect.Current.Count - 1 do
    begin
      if ParName = FQSelect.Current[i].Name then
      begin
        Result := False;
        break;
      end;
    end;
  end;

begin
  if FQSelect.Params.Count > 0 then
  begin
    TempList := TList.Create;
    try
      for x := 0 to FQSelect.Params.Count - 1 do
      begin
        if not CheckCorrectParamName(FQSelect.Params[x].Name) then
          raise Exception.Create(gb_SNotCorrectParamName);
        if Pos('M$S$T$_', FQSelect.Params[x].Name) = 0 then
          TempList.Add(Pointer(x));
      end;
      if TempList.Count > 0 then
      begin
        FSQLParams.Count := TempList.Count;
        for x := 0 to TempList.Count - 1 do
{$IFDEF VER_IBX}
          FSQLParams.AddName(FQSelect.Params[Integer(TempList[x])].Name, x);
{$ELSE}
          THackFIBXSQLDA(FSQLParams).AddName(FQSelect.Params[Integer(TempList[x])].Name, x);
{$ENDIF}
        for x := 0 to TempList.Count - 1 do
        begin
{$IFDEF VER_IBX}
          FSQLParams[x].Assign(FQSelect.Params[Integer(TempList[x])]);
{$ELSE}
          FSQLParams[x].Assign(FQSelect.Params[Integer(TempList[x])]);
          FSQLParams[x].IsNull := True;
{$ENDIF}
        end;
      end;
    finally
      TempList.Free;
    end;
  end;
  FParamsAssigned := True;
end;

procedure TgbDataSet.ConvertSQLs(ReAssign: Boolean);

 procedure ChangePlan(SQL: TStrings);
 var
  P, NextPos: integer;
  S: string;
 begin
   S := SQL.Text;
   NextPos := 0;
   P := GetPosOfStatement(S, 'plan', NextPos);
   if p > 0 then
   begin
     System.Delete(S, P, NextPos - P);
     System.Insert(FPlanForDescQuerys, S, P);
     SQL.Text := S;
   end;
 end;

begin
  inherited;
  if FPlanForDescQuerys <> '' then
  begin
    ChangePlan(FQSelectDesc.SQL);
    ChangePlan(FQSelectDescPart.SQL);
  end;
end;

function TgbDataSet.GetSQLParams: {$IFDEF VER_IBX} TIBXSQLDA {$ELSE} TFIBXSQLDA {$ENDIF};
begin
  if not FParamsAssigned then
  begin
    if not FQSelect.Prepared then ActivateFQSelect;
    AssignParams;
  end;
  Result := FSQLParams;
end;

procedure TgbDataSet.InternalClose;
begin
  FSQLParams.Count := 0;
  FParamsAssigned := False;
  inherited;
  if FUseMacros then
  begin
    if FSQLSaved then
    begin
      FQSelect.SQL.Text := FSavedSQLs[0];
      FInsertText.Text := FSavedSQLs[1];
      FUpdateText.Text := FSavedSQLs[2];
      FDeleteText.Text := FSavedSQLs[3];
      FRefreshText.Text := FSavedSQLs[4];
      FQSelectAlreadyInArray := False;
      FSQLSaved := False;
    end
    else
    begin
      if FQSelectAlreadyInArray then
      begin
        FQSelect.SQL.Text := FSavedSQLs[0];
        FQSelectAlreadyInArray := False;
      end;
    end;
  end;
end;

procedure TgbDataSet.InternalInitFieldDefs;
begin
  if not FQSelect.Prepared then ActivateFQSelect;
  inherited;
end;

function TgbDataSet.ParamByName(Idx: String): {$IFDEF VER_IBX} TIBXSQLVAR {$ELSE} TFIBXSQLVAR {$ENDIF};
begin
  if not FParamsAssigned then
  begin
    if not FQSelect.Prepared then ActivateFQSelect;
    AssignParams;
  end;
{$IFDEF VER_IBX}
  Result := FSQLParams.ByName(Idx);
{$ELSE}
  Result := FSQLParams.ByName[Idx];
{$ENDIF}
end;

procedure TgbDataSet.SetFirstParams;
begin
  if not FParamsAssigned then AssignParams;
  inherited;
end;

procedure TgbDataSet.SQLChanging(Sender: TObject);
begin
  if Active then Close;
end;

function TgbDataSet.MacroByName(Idx: string): TgbMacro;
var
  x: integer;
begin
  for x := 0 to FMacrosDef.FMacros.Count - 1 do
  begin
    if AnsiUpperCase(FMacrosDef.FMacros[x].FMacroName) = AnsiUpperCase(Idx) then
    begin
      Result := FMacrosDef.FMacros[x];
      Exit;
    end;
  end;
  Result := TgbMacro(FMacrosDef.FMacros.Add);
  Result.FMacroName := Idx;
end;

procedure TgbDataSet.InternalOpen;
begin
  if FUseMacros then
  begin
    if not FQSelectAlreadyInArray then
      FSavedSQLs[0] := FQSelect.SQL.Text;
    FSavedSQLs[1] := FInsertText.Text;
    FSavedSQLs[2] := FUpdateText.Text;
    FSavedSQLs[3] := FDeleteText.Text;
    FSavedSQLs[4] := FRefreshText.Text;
    FSQLSaved := True;
    ExpandMacros;
  end;
  inherited;
end;

function TgbDataSet.ExpandSQL(const SQL: string): string;
var
  CurPos, CurEnd: integer;
  CurrentToken, MacValue: string;
  BlankInt: Boolean;
begin
  BlankInt := Database.SQLDialect = 3;
  Result := SQL;
  if Length(Result) > 0 then CurPos := 1 else Exit;
  CurrentToken := TokenNext(Result, BlankInt, CurPos, CurEnd);
  while CurrentToken <> '' do
  begin
    if (CurrentToken[1] = FMacroChar) and (Length(CurrentToken) > 1) then
    begin
      MacValue := MacroByName(Copy(CurrentToken, 2, Length(CurrentToken))).FMacroValue;
      System.Delete(Result, CurPos, CurEnd - CurPos);
      System.Insert(MacValue, Result, CurPos);
      CurEnd := CurPos + Length(MacValue);
    end;
    CurPos := CurEnd;
    CurrentToken := TokenNext(Result, BlankInt, CurPos, CurEnd);
  end;
end;

procedure TgbDataSet.ExpandMacros;
begin
  Include(IdentChars, FMacroChar);
  FQSelect.SQL.Text := ExpandSQL(FSavedSQLs[0]);
  FInsertText.Text := ExpandSQL(FSavedSQLs[1]);
  FUpdateText.Text := ExpandSQL(FSavedSQLs[2]);
  FDeleteText.Text := ExpandSQL(FSavedSQLs[3]);
  FRefreshText.Text := ExpandSQL(FSavedSQLs[4]);
end;

function TgbDataSet.SynchronizeSQL(NumInArray: integer): TStrings;
begin
  FCurrentSQLInSynchr := -1;
  FSynchrList.Text := FSavedSQLs[NumInArray];
  FCurrentSQLInSynchr := NumInArray;
  Result := FSynchrList;
end;

procedure TgbDataSet.SynchrChanged(Sender: TObject);
begin
  if FCurrentSQLInSynchr > -1 then
  begin
    case FCurrentSQLInSynchr of
      0: FQSelect.SQL.Assign(FSynchrList);
      1: FInsertText.Assign(FSynchrList);
      2: FUpdateText.Assign(FSynchrList);
      3: FDeleteText.Assign(FSynchrList);
      4: FRefreshText.Assign(FSynchrList);
    end;
    FCurrentSQLInSynchr := -1;
  end;
end;


{ TgbMacro }

function TgbMacro.GetDisplayName: string;
begin
  Result := 'Macro: ' + FMacroName;
end;

{ TgbMacros }

function TgbMacros.GetMacro(Index: Integer): TgbMacro;
begin
  Result := TgbMacro(inherited Items[Index]);
end;

procedure TgbMacros.SetMacro(Index: Integer; Value: TgbMacro);
begin
  inherited Items[Index] := Value;
end;

{ TMacrosDef }

constructor TMacrosDef.Create(AOwner: TComponent);
begin
  inherited Create;
  FOwner := TgbDataSet(AOwner);
  FMacros := TgbMacros.Create(Self, TgbMacro);
end;

function TMacrosDef.GetMacroChar: Char;
begin
  Result := FOwner.FMacroChar;
end;

function TMacrosDef.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

function TMacrosDef.GetUseMacros: Boolean;
begin
  Result := FOwner.FUseMacros;
end;

procedure TMacrosDef.SetMacroChar(const Value: Char);
begin
  if FOwner.FMacroChar <> Value then
  begin
    if FOwner.Active then FOwner.Close;
    FOwner.FMacroChar := Value;
  end;
end;

procedure TMacrosDef.SetUseMacros(Value: Boolean);
begin
  if FOwner.FUseMacros <> Value then
  begin
    if FOwner.Active then FOwner.Close;
    FOwner.FUseMacros := Value;
  end;
end;


{ TgbBlobStream }

constructor TgbBlobStream.Create(AField: TField; ABlobStream: TgbIBBlobStream; Mode: TBlobStreamMode);
begin
  FModified := False;
  FField := AField;
  FBlobStream := ABlobStream;
  FBlobStream.Seek(0, soFromBeginning);
  if (Mode = bmWrite) then
    FBlobStream.Truncate;
end;

destructor TgbBlobStream.Destroy;
begin
  if FModified then
  begin
    FModified := False;
    if not TBlobField(FField).Modified then
      TBlobField(FField).Modified := True;
    TgbCustomDataSet(FField.DataSet).DataEvent(deFieldChange, Longint(FField));
  end;
  if FBlobStream.Mode <> bmReadWrite then
    FBlobStream.Free;
  inherited Destroy;
end;

function TgbBlobStream.Read(var Buffer; Count: Integer): Longint;
begin
  Result := FBlobStream.Read(Buffer, Count);
end;

function TgbBlobStream.Seek(Offset: Integer; Origin: Word): Longint;
begin
  Result := FBlobStream.Seek(Offset, Origin);
end;

procedure TgbBlobStream.SetSize(NewSize: Integer);
begin
  FBlobStream.SetSize(NewSize);
end;

function TgbBlobStream.Write(const Buffer; Count: Integer): Longint;
begin
  FModified := True;
  if not (FField.DataSet.State in [dsEdit, dsInsert]) then
    raise Exception.Create(gb_SNotInEditMode);
  TgbCustomDataSet(FField.DataSet).SetModified(True);
  Result := FBlobStream.Write(Buffer, Count);
end;


end.
