{********************************************************}
{                                                        }
{       InterBase EventAlerter components                }
{       Copyright (c) 1995 Borland International         }
{                                                        }
{       Written by:                                      }
{         James Thorpe                                   }
{         CSA Australasia                                }
{         Compuserve: 100035,2064                        }
{         Internet:   csa@csaa.com.au                    }
{                                                        }
{       Adapted for Delphi 1.02 by                       }
{         Oleg Kukartsev                                 }
{       E-mail: Please look at InterBase List Server     }
{               interbase@esunix1.emporia.edu            }
{                                                        }
{********************************************************}

{$Ifdef Win32}
  {.$Define WaitForThreadTermination}
{$Endif}

unit IBCtrls;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DB, IBProc;

const
  MaxEvents = 15;
  EventLength = 64;

type
  TIBComponent = class(TComponent)
  private
    FDatabase: TDatabase;
    procedure SetDatabase(Value: TDatabase);
    procedure ValidateDatabase(ADatabase: TDatabase);
  protected
    function GetNativeHandle: isc_db_handle;
    procedure HandleIBErrors(status: pstatus_vector);
    function IsInterbaseDatabase(ADatabase: TDatabase): Boolean;
  published
    property  Database: TDatabase read FDatabase write SetDatabase;
  end;

  TEventAlert = procedure(Sender: TObject; EventName: string; EventCount: longint;
                           var CancelAlerts: Boolean) of object;

  TEventBuffer = array[0..MaxEvents-1, 0..EventLength-1] of Char;

  TIBEventAlerter = class(TIBComponent)
  private
    LibHandle: THandle;
    {$Ifdef Win32}
      EventThreadHandle : THandle;
      CS: TRTLCriticalSection;
    {$Endif}
    FEvents: TStrings;
    FOnEventAlert: TEventAlert;
    FQueued: Boolean;
    FRegistered: Boolean;
    Changing: Boolean;
    EventBuffer: PChar;
    EventBufferLen: integer;
    EventID: isc_long;
    ProcessingEvents: Boolean;
    RegisteredState: Boolean;
    ResultBuffer: PChar;
    FWindowHandle: HWND;
    procedure DoQueueEvents;
    procedure EventChange(Sender: TObject);
    procedure WndProc(var Msg: TMessage);
  protected
    procedure HandleEvent;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure SetEvents(Value: TStrings);
    procedure SetDatabase(Value: TDatabase);
    procedure SetRegistered(Value: boolean);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure CancelEvents;
    procedure QueueEvents;
    procedure RegisterEvents;
    procedure UnRegisterEvents;
    property  Queued: Boolean read FQueued;
  published
    property Events: TStrings read FEvents write SetEvents;
    property Registered: Boolean read FRegistered write SetRegistered;
    property OnEventAlert: TEventAlert read FOnEventAlert write FOnEventAlert;
  end;

  EIBError = class(Exception);

implementation
uses
  DbiTypes, DbiProcs, DbiErrs;

{$Ifndef Win32}
  type
    ShortString = String;
{$Endif}

const
  SNoEventsRegistered  =
    'You must register events before queueing them';
  SInvalidDBConnection =
    'Component is not connected to an open Database';
  SInvalidDatabase     =
    '''%s'' is not connected to an InterBase database';
  SInvalidCancellation =
    'You cannot call CancelEvents from within an OnEventAlert handler';
  SInvalidEvent        =
    'Invalid blank event added to EventAlerter events list';
  SInvalidQueueing     =
    'You cannot call QueueEvents from within an OnEventAlert handler';
  SInvalidRegistration =
    'You cannot Register or Unregister events from within an OnEventAlert handler';
  SMaximumEvents       =
    'You can only register 15 events per EventAlerter';

  {$Ifdef Win32}
    LibraryName = 'gds32.dll';
  {$Else}
    LibraryName = 'gds.dll';
  {$Endif}

var
  { Dynamically Loaded InterBase API functions (gds[32].dll) }
  IscQueEvents: TIscQueEvents;
  IscFree: TIscfree;
  IscEventBlock: TIscEventBlock;
  IscEventCounts: TIscEventCounts;
  IscCancelEvents: TIscCancelEvents;
  IscInterprete: TIscInterprete;

{ TIBComponent }
function AssignedAndConnected(Database : TDatabase) : Boolean;
begin
  Result := Assigned(Database) and Database.Connected;
end;

function TIBComponent.GetNativeHandle: isc_db_handle;
var
  length: word;
begin
  if AssignedAndConnected(FDatabase) then
    Check(DbiGetProp(HDBIOBJ(FDatabase.Handle), dbNATIVEHNDL,
      @Result, sizeof(isc_db_handle), length))
  else Result := nil;
end;

procedure TIBComponent.HandleIBErrors(status: pstatus_vector);
var
  Buffer: array[0..255] of char;
  errMsg, lastMsg: ShortString;
  ErrCode: isc_status;
begin
  errMsg := '';
  repeat
    ErrCode := IscInterprete(@Buffer, @status);
    if lastMsg <> strPas(Buffer) then begin
      lastMsg := strPas(Buffer);
      if length(errMsg) <> 0 then errMsg := errMsg+#13#10;
      errMsg := errMsg+lastMsg;
    end;
  until ErrCode = 0;
  raise EIBError.Create(errMsg);
end;

function TIBComponent.IsInterbaseDatabase(ADatabase: TDatabase): Boolean;
var
  Length: Word;
  Buffer: array[0..63] of Char;
begin
  Result := False;
  if ADatabase.Handle <> nil then begin
    Check(DbiGetProp(HDBIOBJ(ADatabase.Handle), dbDATABASETYPE,
      @Buffer, SizeOf(Buffer), Length));
    Result := StrIComp(Buffer, 'INTRBASE') = 0;
  end;
end;

procedure TIBComponent.SetDatabase(Value: TDatabase);
begin
  if Value <> FDatabase then begin
    if AssignedAndConnected(Value) then ValidateDatabase(Value);
    FDatabase := Value;
  end;
end;

procedure TIBComponent.ValidateDatabase(ADatabase: TDatabase);
begin
  if not AssignedAndConnected(ADatabase) then
    raise EIBError.Create(SInvalidDBConnection)
  else if not IsInterbaseDatabase(ADatabase) then
    raise EIBError.CreateFmt(SInvalidDatabase, [ADatabase.Name]);
end;

{ TIBEventAlerter }
procedure IBEventCallback(IBEventAlerter: TIBEventAlerter;
  BufferLength: short; UpdatedBuffer: PChar); cdecl; export;
begin
  if Assigned(IBEventAlerter) and (BufferLength > 0) then
    with IBEventAlerter do begin
      {$Ifdef Win32}
        if Changing then begin
          if EventThreadHandle <> 0 then
            CloseHandle(EventThreadHandle);
          DuplicateHandle(GetCurrentProcess, GetCurrentThread, GetCurrentProcess,
            @EventThreadHandle, SYNCHRONIZE or 1{THREAD_TERMINATE}, False, 0);
        end; { if Changing }
        try
          EnterCriticalSection(CS);
          Move(UpdatedBuffer^, ResultBuffer^, BufferLength);
        finally
          LeaveCriticalSection(CS);
        end;
        PostMessage(FWindowHandle, WM_User + 1, 0, 0);
      {$Else}
        Move(UpdatedBuffer^, ResultBuffer^, BufferLength);
        PostMessage(FWindowHandle, WM_User + 1, 0, 0);
      {$Endif}
    end; { with IBEventAlerter }
end;

procedure TIBEventAlerter.WndProc(var Msg: TMessage);
begin
  with Msg do begin
    if Msg = WM_User + 1 then begin
      try
        HandleEvent;
      except
        Application.HandleException(Self);
      end;
      Result := 1;
    end
    {else Result := 0;}
    else if Msg = WM_ActivateApp then Result := 0
    else Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
  end;
end;

function GetProcAddressX(ALibHandle: THandle; IscFunctionName : ShortString) : TFarProc;
begin
  {$Ifdef Win32}
    IscFunctionName := IscFunctionName + #0;
  {$Else}
    IscFunctionName := '_' + IscFunctionName + #0;
  {$Endif}
  Result := GetProcAddress(ALibHandle, @IscFunctionName[1]);
  if Result = nil then
    raise EDLLLoadError.Create('Failed to lookup ' + IscFunctionName);
end;

constructor TIBEventAlerter.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FWindowHandle := AllocateHWnd(WndProc);
  FEvents := TStringList.Create;
  with TStringList(FEvents) do begin
    OnChange := EventChange;
    Duplicates := dupIgnore;
  end;
  { Attempt to load GDS[32].DLL. If this fails then raise an exception. }
  { This will cause the component not to be created }
  LibHandle := LoadLibrary(LibraryName);
  if LibHandle < 32 then
    raise EDLLLoadError.Create('Unable to load ' + LibraryName);

  @IscQueEvents    := GetProcAddressX(LibHandle, 'isc_que_events');
  @IscInterprete   := GetProcAddressX(LibHandle, 'isc_interprete');
  @IscFree         := GetProcAddressX(LibHandle, 'isc_free');
  @IscEventBlock   := GetProcAddressX(LibHandle, 'isc_event_block');
  @IscEventCounts  := GetProcAddressX(LibHandle, 'isc_event_counts');
  @IscCancelEvents := GetProcAddressX(LibHandle, 'isc_cancel_events');

  {$Ifdef Win32}
    InitializeCriticalSection(CS);
  {$Endif}
end;

destructor TIBEventAlerter.Destroy;
begin
  UnregisterEvents;
  {$Ifdef Win32}
    if EventThreadHandle <> 0 then begin
      {$Ifdef WaitForThreadTermination}
        WaitForSingleObject(EventThreadHandle, 10000{Infinite});
      {$Else}
        TerminateThread(EventThreadHandle, 0);
      {$Endif}
      CloseHandle(EventThreadHandle);
    end;
    DeleteCriticalSection(CS);
  {$Endif}

  if LibHandle >= 32 then FreeLibrary(LibHandle);
  FEvents.Free;
  DeallocateHWnd(FWindowHandle);
  inherited Destroy;
end;

procedure TIBEventAlerter.CancelEvents;
var
  status: status_vector;
  ErrCode: isc_status;
  dbHandle: isc_db_handle;
begin
  if FQueued then begin
    if ProcessingEvents then
      raise EIBError.Create(SInvalidCancellation);
    {$Ifdef Win32}
      try
        EnterCriticalSection(CS);
        if AssignedAndConnected(FDatabase) then begin
          {Changing := True;}
          dbHandle := GetNativeHandle;
          ErrCode := IscCancelEvents(@status, @dbHandle, @EventID);
          if ErrCode <> 0 then HandleIBErrors(@status)
        end;
      finally
        LeaveCriticalSection(CS);
      end;
    {$Else}
      if AssignedAndConnected(FDatabase) then begin
        {Changing := True;}
        dbHandle := GetNativeHandle;
        ErrCode := IscCancelEvents(@status, @dbHandle, @EventID);
        if ErrCode <> 0 then HandleIBErrors(@status)
      end;
    {$Endif}
    FQueued := False;
  end;
end;

procedure TIBEventAlerter.DoQueueEvents;
var
  status: status_vector;
  ErrCode: isc_status;
  dbHandle: isc_db_handle;
begin
  {$Ifdef Win32}
    try
      EnterCriticalSection(CS);
      dbHandle := GetNativeHandle;
      ErrCode := IscQueEvents(@status, @dbHandle, @EventID, EventBufferLen,
                                EventBuffer, @IBEventCallback, Self);
      if ErrCode <> 0 then HandleIBErrors(@status);
    finally
      LeaveCriticalSection(CS);
    end;
  {$Else}
    ValidateDatabase(DataBase);
    dbHandle := GetNativeHandle;
    ErrCode := IscQueEvents(@status, @dbHandle, @EventID, EventBufferLen,
                              EventBuffer, @IBEventCallback, Self);
    if ErrCode <> 0 then HandleIBErrors(@status);
  {$Endif}
  FQueued := True;
end;

procedure TIBEventAlerter.EventChange(Sender: TObject);
begin
  if TStringList(Events).IndexOf('') <> -1 then { check for blank event }
    raise EIBError.Create(SInvalidEvent);
  if Events.Count > MaxEvents then begin        { check for too many events }
    TStringList(Events).OnChange := nil;
    Events.Delete(MaxEvents);
    TStringList(Events).OnChange := EventChange;
    raise EIBError.Create(SMaximumEvents);
  end;
  if Registered then RegisterEvents;
end;

procedure TIBEventAlerter.HandleEvent;
var
  CancelAlerts: Boolean;
  i: integer;
  status: status_vector;
begin
  try
    { prevent modification of vital data structures while handling events }
    ProcessingEvents := True;
    {$Ifdef Win32}
      try
        EnterCriticalSection(CS);
        IscEventCounts(@status, EventBufferLen, EventBuffer, ResultBuffer);
      finally
        LeaveCriticalSection(CS);
      end;
    {$Else}
      IscEventCounts(@status, EventBufferLen, EventBuffer, ResultBuffer);
    {$Endif}
    CancelAlerts := False;
    if Assigned(FOnEventAlert) and not Changing then begin
      for i := 0 to Events.Count-1 do begin
        if (status[i] <> 0) and not CancelAlerts then
          FOnEventAlert(Self, Events[Events.Count-i-1], status[i], CancelAlerts);
      end;
    end;
    Changing := False;
    if not CancelAlerts and FQueued then DoQueueEvents;
  finally
    ProcessingEvents := False;
  end;
end;

procedure TIBEventAlerter.Loaded;
begin
  inherited Loaded;
  try
    if RegisteredState then RegisterEvents;
  except
    if csDesigning in ComponentState then
      Application.HandleException(self)
    else raise;
  end;
end;

procedure TIBEventAlerter.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FDatabase) then begin
    UnregisterEvents;
    FDatabase := nil;
  end;
end;

procedure TIBEventAlerter.QueueEvents;
begin
  if not FRegistered then
    raise EIBError.Create(SNoEventsRegistered);
  if ProcessingEvents then
    raise EIBError.Create(SInvalidQueueing);
  if not FQueued then begin
    ValidateDatabase(DataBase);
    Changing := True;
    DoQueueEvents;
  end;
end;

procedure TIBEventAlerter.RegisterEvents;
var
  i: integer;
  bufptr: pointer;
  eventbufptr: pointer;
  resultbufptr: pointer;
  buflen: integer;
  Buffer: TEventBuffer;
begin
  ValidateDatabase(Database);
  if csDesigning in ComponentState then FRegistered := True
  else begin
    UnregisterEvents;
    if Events.Count = 0 then Exit;
    for i := 0 to Events.Count-1 do
      StrPCopy(@Buffer[i][0], Events[i]);
    i := Events.Count;
    bufptr := @buffer[0];
    eventbufptr :=  @EventBuffer;
    resultBufPtr := @ResultBuffer;
    asm
    {$Ifdef Win32}
       mov     ecx, dword ptr [i]
       mov     eax, dword ptr [bufptr]

     @@Next:
       push    eax
       add     eax, EventLength
       loop    @@Next

       push    dword ptr [i]
       push    dword ptr [resultBufPtr]
       push    dword ptr [eventBufPtr]
       call    [IscEventBlock]
       mov     dword ptr [bufLen], eax
       mov     eax, dword ptr [i]
       shl     eax, 2
       add     eax, 12
       add     esp, eax
     {$Else}
       mov     cx, word ptr [i]
       mov     ax, word ptr [bufptr]
       mov     dx, word ptr [bufptr+2]

     @@Next:
       push    dx
       push    ax
       add     ax, EventLength
       loop    @@Next

       push    word ptr [i]
       push    word ptr [resultBufPtr+2]
       push    word ptr [resultBufPtr]
       push    word ptr [EventBufPtr+2]
       push    word ptr [EventBufPtr]
       call    [IscEventBlock]
       mov     word ptr [bufLen], ax
       mov     ax, word ptr [i]
       shl     ax, 2
       add     ax, 10
       add     sp, ax
     {$Endif}
    end;
    EventBufferlen := Buflen;
    FRegistered := True;
    QueueEvents;
  end;
end;

procedure TIBEventAlerter.SetEvents(Value: TStrings);
begin
  FEvents.Assign(Value);
end;

procedure TIBEventAlerter.SetDatabase(Value: TDatabase);
begin
  if Value <> FDatabase then begin
    UnregisterEvents;
    inherited SetDatabase(Value);
  end;
end;

procedure TIBEventAlerter.SetRegistered(Value: Boolean);
begin
  if (csReading in ComponentState) then
    RegisteredState := Value
  else if FRegistered <> Value then
    if Value then RegisterEvents else UnregisterEvents;
end;

procedure TIBEventAlerter.UnregisterEvents;
begin
  if Assigned(Self) then begin
    if ProcessingEvents then
      raise EIBError.Create(SInvalidRegistration);

    if csDesigning in ComponentState then FRegistered := False
    else if not (csLoading in ComponentState) then begin
      CancelEvents;
      if FRegistered then begin
        IscFree(EventBuffer);
        EventBuffer := nil;
        IscFree(ResultBuffer);
        ResultBuffer := nil;
      end;
      FRegistered := False;
    end;
  end;
end;

end.


