
{ Need to refer to a database component. This is done via a datasource }
{ which is eventually connected to a table/query which has a database }
{ it uses. }

{ When you set up DataSource, TableName is either blanked, or set to }
{ the name that the TTable uses, if the DataSource refers to a TTable }

{ Same problems happen with other data aware components if the table }
{ is created after the data aware control. Use Edit | Creation Order }
{ to ensure things are set up right }
unit IBArray;
{$define ReadOnly}

interface

uses
  Classes, DB, BDE, IBProc, DBTables;

type
  EInterBaseError = class(EDatabaseError);

  TEditingChangeEvent = procedure(Sender: TObject; Editing: Boolean) of object;

  TInterBase = class(TComponent)
  private
    FDatabase: TDatabase;
    procedure SetDatabase(Value: TDatabase);
    procedure ValidateDatabase(Database: TDatabase);
  protected
    function GetNativeHandle: IscDbHandle;
    function IsInterbaseDatabase(Database: TDatabase): Boolean;
    property Database: TDatabase read FDatabase write SetDatabase;
  end;

  TIBArray = class(TInterBase)
  private
    { Property fields }
    FDataLink: TFieldDataLink;
    FTableName: String;
    { Triggered when value changes, usually by row change or post }
    FOnDataChange: TNotifyEvent;
{$ifndef ReadOnly}
    { Triggered when editing state changes }
    FOnEditingChange: TEditingChangeEvent;
    { Triggered when a record is about to be posted }
    FOnUpdateData: TNotifyEvent;
{$endif}
    { Internal fields }
    FFieldSetup: Boolean;
    FTranHdl: IscTrHandle;
    FDBHdl: IscDbHandle;
    FStatus: TStatusVector;
    FArrayDesc: TIscArrayDesc;
    FArrayId: TIscQuad;
    FDataSize: Longint;
    FDatumSize: Word;
    FDatumType: TFieldType;
    FDimensionCount: Smallint;
    FInTransaction: Boolean;
  protected
    { Property readers and writers }
    function GetDataField: String;
    function GetDataSource: TDataSource;
    function GetDimension(Index: Integer): TIscArrayBound;
    function GetField: TField;
{$ifndef ReadOnly}
    function GetReadOnly: Boolean;
{$endif}
    procedure SetDataField(const Value: String);
    procedure SetDataSource(Value: TDataSource);
{$ifndef ReadOnly}
    procedure SetReadOnly(Value: Boolean);
{$endif}
    { Internal routines }
    { Called when field info changes, e.g. on record scroll }
    procedure DataChange(Sender : TObject);
    procedure DoTransaction(Start: Boolean);
{$ifndef ReadOnly}
    { Called when Editing state toggles }
    procedure EditingChange(Sender : TObject);
{$endif}
    procedure SetupFieldProperties;
{$ifndef ReadOnly}
    { Triggered by a call to TDBDataSet.UpdateRecord }
    procedure UpdateData(Sender : TObject);
{$endif}
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
{$ifndef ReadOnly}
    procedure Clear;
{$endif}
    { Fill buffer with field data. Buffer must be large enough }
    { (use DataSze). If the field is null, no data is transferred, }
    { and the function retuns False, otherwise it returns True }
    function GetData(Buffer: Pointer): Boolean;
    { Obtain native IB handle, and set Database property }
    function GetInterBaseHandle: IscDbHandle;
    { Used to set DataSource to nil, if someon deletes the data source component }
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
{$ifndef ReadOnly}
    { Takes buffer and fills field with its data. Buffer must be }
    { the right size (use DataSize). If Buffer = nil, the field }
    { will be null }
    procedure SetData(Buffer: Pointer);
{$endif}
    { Size of complete array }
    property DataSize: Longint read FDataSize;
    { Array element size }
    property DatumSize: Word read FDatumSize;
    { Array element type }
    property DatumType: TFieldType read FDatumType;
    { Number of dimensions }
    property DimensionCount: Smallint read FDimensionCount;
    { Dimensions sizes }
    property Dimensions[Index: Integer]: TIscArrayBound read GetDimension; default;
    { The Delphi field object representing the array - it is a TBytesField }
    property Field: TField read GetField;
  published
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DataField: String read GetDataField write SetDataField;
{$ifndef ReadOnly}
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly;
{$endif}
    property TableName: String read FTableName write FTableName;
    { Triggered when Editing state toggles }
    property OnDataChange: TNotifyEvent read FOnDataChange write FOnDataChange;
{$ifndef ReadOnly}
    property OnEditingChange: TEditingChangeEvent read FOnEditingChange write FOnEditingChange;
    property OnUpdateData: TNotifyEvent read FOnUpdateData write FOnUpdateData;
{$endif}
  end;

{ Tests for IB errors and generates exception if necessary }
procedure IBCheck(var Status: TStatusVector);

{ Data type conversions: }

{ Varying(x) takes x + 2 bytes. The string is terminated by #10#0 }

function SQLVaryingToString(Source: PChar): String;

procedure StringToSQLVarying(Dest: PChar; Source: String; MaxLen: Cardinal);

procedure Register;

implementation

uses
  SysUtils, Forms, IBDesign, DsgnIntf, DBConsts, Consts;

procedure IBCheck(var Status: TStatusVector);
var
  Buffer: array[0..255] of Char;
  ErrMsg, LastMsg: String;
  ErrCode: IscStatus;
  StatusAddr: PStatusVector;
begin
  if (Status[0] = 1) and (Status[1] > 0) then
  begin
    ErrMsg := '';
    StatusAddr := @Status;
    repeat
      ErrCode := IscInterprete(Buffer, StatusAddr);
      if LastMsg <> StrPas(Buffer) then
      begin
        LastMsg := StrPas(Buffer);
        if Length(ErrMsg) <> 0 then
          AppendStr(ErrMsg, #13#10);
        ErrMsg := ErrMsg + LastMsg;
      end;
    until ErrCode = 0;
    raise EInterBaseError.Create(ErrMsg);
  end;
end;

function SQLVaryingToString(Source: PChar): String;
begin
  Result := String(Source);
  { Remove trailing #10 }
  if Length(Result) > 0 then
    SetLength(Result, Pred(Length(Result)));
end;

procedure StringToSQLVarying(Dest: PChar; Source: String; MaxLen: Cardinal);
begin
  if MaxLen >= 2 then
    StrLCopy(Dest, PChar(Source), MaxLen - 2);
  StrCat(Dest, #10);
end;

procedure TInterBase.SetDatabase(Value: TDatabase);
begin
  if Value <> FDatabase then
  begin
    if Assigned(Value) and Value.Connected then
      ValidateDatabase(Value);
    FDatabase := value;
  end;
end;

procedure TInterBase.ValidateDatabase(Database: TDatabase);
begin
  if not Assigned(Database) or not Database.Connected then
    raise EInterBaseError.CreateFmt(
      '''%s'' is not connected to an open Database', [Name]);
  if not IsInterbaseDatabase(Database) then
    raise EInterBaseError.CreateFmt(
      '''%s'' is not connected to an InterBase database', [Name]);
end;

function TInterBase.GetNativeHandle: IscDbHandle;
var
  Length: Word;
begin
  Result := nil;
  if Assigned(Database) and FDatabase.Connected then
    Check(DbiGetProp(HDbiObj(FDatabase.Handle), dbNativeHndl,
      @Result, sizeof(IscDbHandle), Length));
end;

function TInterBase.IsInterbaseDatabase(Database: TDatabase): Boolean;
var
  Length: Word;
  Buffer: array[0..63] of Char;
begin
  Result := False;
  if Assigned(Database.Handle) then
  begin
    Check(DbiGetProp(HDbiObj(Database.Handle), dbDataBaseType, @Buffer,
      SizeOf(Buffer), Length));
    Result := StrIComp(Buffer, 'INTRBASE') = 0;
  end;
end;

constructor TIBArray.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDataLink := TFieldDataLink.Create;
  FDataLink.OnDataChange := DataChange;
{$ifndef ReadOnly}
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
{$endif}
end;

destructor TIBArray.Destroy;
begin
  FDataLink.Free;
  inherited Destroy;
  DoTransaction(False);
end;

function TIBArray.GetDataField: String;
begin
  Result := FDataLink.FieldName;
end;

procedure TIBArray.SetDataField(const Value: String);
begin
  if (FTableName = '') and (Value <> '') and
     not(csLoading in ComponentState) then
    raise EInterBaseError.Create('Must set TableName property first');
  FFieldSetup := False;
  FDataLink.FieldName := Value;
  { Don't try and connect whilst form is loading - databases aren't open }
  if not (csLoading in ComponentState) then
    try
      SetupFieldProperties;
    except
      on E: EInterBaseError do
        E.Message := Value + ' is not an array';
    end;
end;

function TIBArray.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TIBArray.SetDataSource(Value: TDataSource);
begin
  if Value <> DataSource then
  begin
    if not (csLoading in ComponentState) then
      if Assigned(Value) and (Value.DataSet is TTable) then
        FTableName := TTable(Value.DataSet).TableName
      else
        FTableName := '';
    FDataLink.DataSource := Value;
  end;
end;

function TIBArray.GetDimension(Index: Integer): TIscArrayBound;
begin
  if not FFieldSetup then
    SetupFieldProperties;
  if not (Index in [0..FArrayDesc.Dimensions]) then
    raise EListError.CreateRes(SListIndexError);
  Result := FArrayDesc.Bounds[Index];
end;

function TIBArray.GetField: TField;
begin
  Result := nil;
  if Assigned(FDataLink) then
    Result := FDataLink.Field;
end;

{$ifndef ReadOnly}
function TIBArray.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TIBArray.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;
{$endif}

procedure TIBArray.SetupFieldProperties;
var
  Loop: Byte;
begin
  if DataField = '' then
  begin
    FDataSize := 0;
    FDatumSize := 0;
    FDatumType := ftUnknown;
  end
  else
  begin
    DoTransaction(True);
    IscArrayLookupBounds(FStatus, FDBHdl, FTranHdl,
      PChar(FTableName), PChar(DataField), FArrayDesc);
    IBCheck(FStatus);
    with FArrayDesc do
    begin
      case DType of
        BlrText, BlrVarying: FDatumType := ftString;
        BlrShort: FDatumType := ftSmallint;
        BlrLong: FDatumType := ftInteger;
        BlrFloat, BlrDouble, BlrD_Float: FDatumType := ftFloat;
        BlrDate: FDatumType := ftDateTime;
      end;
      FDatumSize := Length;
      { Varying text fields have a #10#0 terminator }
      if DType = BlrVarying then
        Inc(FDatumSize, 2);
      FDimensionCount := Dimensions;
      FDataSize := 0;
      for Loop := 0 to Dimensions - 1 do
        Inc(FDataSize, FDatumSize * (Bounds[Loop].Upper - Bounds[Loop].Lower + 1));
    end;
    FFieldSetup := True;
  end;
end;

procedure TIBArray.DoTransaction(Start: Boolean);
var
  Teb: TIscTeb;
begin
  if Start and not FInTransaction then
  begin
    { Get current database handle }
    FDBHdl := GetInterBaseHandle;
    { Start a transaction }
    FTranHdl := nil;
    Teb.DBPtr := @FDBHdl;
    Teb.TPBLen := 0;
    Teb.TPBPtr := nil;
    { Start transaction }
    IscStartMultiple(FStatus, FTranHdl, 1, Teb);
    IBCheck(FStatus);
    FInTransaction := not FInTransaction;
  end
  else if not Start and FInTransaction then
  begin
    { Stop transaction }
    IscCommitTransaction(FStatus, FTranHdl);
    FTranHdl := nil;
    IBCheck(FStatus);
    FInTransaction := not FInTransaction;
  end;
end;

{$ifndef ReadOnly}
procedure TIBArray.Clear;
begin
  SetData(nil);
end;
{$endif}

procedure TIBArray.DataChange(Sender: TObject);
begin
  if Assigned(FDataLink.Field) and
     not (FDataLink.DataSet.State in [dsEdit, dsInsert]) and
     Assigned(FOnDataChange) then
    FOnDataChange(Self);
end;

{$ifndef ReadOnly}
procedure TIBArray.EditingChange(Sender: TObject);
begin
  if Assigned(FDataLink.Field) and Assigned(FOnEditingChange) then
    FOnEditingChange(Self, FDataLink.Editing);
end;

procedure TIBArray.UpdateData(Sender : TObject);
begin
  if Assigned(FOnUpdateData) then
    FOnUpdateData(Self);
end;
{$endif}

procedure TIBArray.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and Assigned(FDataLink) and
     (AComponent = DataSource) then
  begin
    DataSource := nil;
    FFieldSetup := False;
  end;
end;

function TIBArray.GetInterBaseHandle: IscDbHandle;
begin
  if (Database = nil) and Assigned(DataSource) and
     Assigned(DataSource.DataSet) then
    Database := TDBDataSet(DataSource.DataSet).Database;
  Result := GetNativeHandle;
end;

function TIBArray.GetData(Buffer: Pointer): Boolean;
var
  LenL: IscLong;
begin
  Result := False;
  if not(csLoading in ComponentState) then
  begin
    { Start transaction }
    DoTransaction(True);
    if not FFieldSetup then
      SetupFieldProperties;
    { Get array slice, checking for null arrays }
    if Field.GetData(@FArrayId) then
    begin
      LenL := FDataSize;
      IscArrayGetSlice(FStatus, FDBHdl, FTranHdl, FArrayId, FArrayDesc, Buffer^, LenL);
      IBCheck(FStatus);
      Result := True;
    end;
    { Stop transaction }
    DoTransaction(False);
  end;
end;

{$ifndef ReadOnly}
procedure TIBArray.SetData(Buffer: Pointer);
var
  LenL: IscLong;
begin
  if not Assigned(Field) then
    raise EInterBaseError.CreateFmt('%s has no field property', [Name]);
  if not Assigned(DataSource) then
    raise EInterBaseError.CreateFmt('%s has no datasource', [Name]);
  if not Assigned(DataSource.DataSet) then
    raise EInterBaseError.CreateFmt('%s has no dataset', [DataSource.Name]);
  if not (DataSource.DataSet.State in [dsInsert, dsEdit]) then
    DBError(SNotEditing);
  if ReadOnly then
    DBErrorFmt(SFieldReadOnly, [DataField]);
  { Start transaction }
  DoTransaction(True);
  if not FFieldSetup then
    SetupFieldProperties;
  { Put array slice, unless its a null }
  Field.GetData(@FArrayId);
  LenL := FDataSize;
  IscArrayPutSlice(FStatus, FDBHdl, FTranHdl, FArrayId, FArrayDesc, Buffer^, LenL);
  IBCheck(FStatus);
  { Stop transaction }
  DoTransaction(False);
  { Need to write new array id to field - mus tbe in edit mode }
  Field.SetData(@FArrayId);
end;
{$endif}

procedure Register;
begin
  RegisterComponents('Samples', [TIBArray]);
  RegisterPropertyEditor(TypeInfo(String), TIBArray,
    'TableName', TIBTableNameProperty);
end;

end.
