
{*******************************************************}
{                                                       }
{       InterBase Connections Service Version 2.3       }
{                                                       }
{                                                       }
{          Copyright (c) 1999-2004 Vadim Crits          }
{                                                       }
{*******************************************************}

unit SvcMain;

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Forms, Menus, ActnList,
  ComCtrls, ImgList, SvcMgr, ShellAPI, UdpBase;

const
  WM_IBCONNSVCICON = WM_USER + 1;
  SPY_TIMEOUT = 13000;

type
  TIBConnForm = class(TForm)
    ActionList: TActionList;
    actStart: TAction;
    actStop: TAction;
    PopupMenu1: TPopupMenu;
    actOpen: TAction;
    piOpen: TMenuItem;
    actExit: TAction;
    N2: TMenuItem;
    piExit: TMenuItem;
    N1: TMenuItem;
    piStart: TMenuItem;
    piStop: TMenuItem;
    StatusBar: TStatusBar;
    lvConnections: TListView;
    ImageList: TImageList;
    actCloseConnection: TAction;
    PopupMenu2: TPopupMenu;
    piCloseConnection: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormDestroy(Sender: TObject);
    procedure lvConnectionsColumnClick(Sender: TObject; Column: TListColumn);
    procedure lvConnectionsCompare(Sender: TObject; Item1, Item2: TListItem;
      Data: Integer; var Compare: Integer);
    procedure lvConnectionsKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure actStartExecute(Sender: TObject);
    procedure actStopExecute(Sender: TObject);
    procedure actOpenExecute(Sender: TObject);
    procedure actExitExecute(Sender: TObject);
    procedure actCloseConnectionExecute(Sender: TObject);
    procedure actCloseConnectionUpdate(Sender: TObject);
  private
    { Private declarations }
    FTaskMessage: Cardinal;
    FFromApp: Boolean;
    FAscending: Boolean;
    FSortCol: Integer;
    FNotifyIconData: TNotifyIconData;
    procedure UpdateStatus;
  protected
    procedure WndProc(var Message: TMessage); override;
    procedure WMIBConnSvcIcon(var Message: TMessage); message WM_IBCONNSVCICON;
    procedure AddIcon;
    procedure ModifyIcon;
  public
    { Public declarations }
    procedure Initialize(FromApp: Boolean);
  end;

{ TIBConnService class }

  TIBConnService = class(TService)
  protected
    procedure Start(Sender: TService; var Started: Boolean);
    procedure Stop(Sender: TService; var Stopped: Boolean);
  public
    function GetServiceController: TServiceController; override;
    constructor CreateNew(AOwner: TComponent; Dummy: Integer = 0); override;
  end;

  PIBConnRow = ^TIBConnRow;
  TIBConnRow = packed record
    dwState: DWORD;
    dwLocalAddr: DWORD;
    dwLocalPort: DWORD;
    dwRemoteAddr: DWORD;
    dwRemotePort: DWORD;
    HostName: string[255];
    dtConnect: TDateTime;
    dtDisconnect: TDateTime;
    dwUId: DWORD;
  end;

{ TIBConnThread class }

  TIBConnThread = class(TThread)
  private
    FIBConnTable: TList;
    FCurRow: PIBConnRow;
    FUId: DWORD;
  protected
    procedure Execute; override;
    procedure AddConnect;
    procedure RemoveConnect;
    function GetUId: DWORD;
  public
    constructor Create(CreateSuspended: Boolean);
    destructor Destroy; override;
  end;

{ TIBSpyThread class}

  TIBSpyThread = class(TThread)
  protected
    procedure Execute; override;
  end;

{ TSvrUdpThread class }

  TSvrUdpThread = class(TCustomUdpThread)
  protected
    procedure Read; override;
  end;

var
  IBConnForm: TIBConnForm;
  IBConnService: TIBConnService;
  IBConnThread: TIBConnThread;
  IBSpyThread: TIBSpyThread;
  SvrUdpThread: TSvrUdpThread;

implementation

uses CSConst, WinSock, IpHlpApi, Registry;

{$R *.dfm}

var
  IBPort: DWORD;
  TimeOut: DWORD;
  LogFile: string;
  SingleLine: Boolean;
  ComputerName: array[0..MAX_COMPUTERNAME_LENGTH] of Char;

{ TIBConForm }

procedure TIBConnForm.WndProc(var Message: TMessage);
begin
  if Message.Msg = FTaskMessage then
    AddIcon;
  inherited WndProc(Message);
end;

procedure TIBConnForm.FormCreate(Sender: TObject);
var
  StartupDir: string;
  Position: TRect;
  dwSize: DWORD;
begin
  with TStringList.Create do
  try
    StartupDir := Copy(ParamStr(0), 1, LastDelimiter(PathDelim, ParamStr(0)));
    try
      LoadFromFile(StartupDir + SCfgFile);
    except
      on E: Exception do
      begin
        MessageBox(0, PChar(E.Message), PChar(SIBConnSvcTitle), MB_OK or MB_ICONERROR);
        Halt;
      end;
    end;
    IBPort := ntohs(StrToInt(Values[SPort]));
    LogFile := Values[SLogFile];
    TimeOut := StrToInt(Values[STimeOut]);
    SingleLine := StrToBool(Values[SSingleLine]);
  finally
    Free;
  end;
  with TRegistry.Create do
  try
    if OpenKey(KEY_IBCONNSVC, False) then
    try
      if ValueExists(SPosition) then
      begin
        ReadBinaryData(SPosition, Position, SizeOf(Position));
        BoundsRect := Position;
      end;
    finally
      CloseKey;
    end;
  finally
    Free;
  end;
  dwSize := SizeOf(ComputerName);
  GetComputerName(ComputerName, dwSize);
  Caption := SIBConnSvcTitle;
  FAscending := False;
  FSortCol := -1;
  UpdateStatus;
end;

procedure TIBConnForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  Hide;
  Action := caNone;
end;

procedure TIBConnForm.FormDestroy(Sender: TObject);
var
  Position: TRect;
begin
  with TRegistry.Create do
  try
    if OpenKey(KEY_IBCONNSVC, True) then
    try
     Position := BoundsRect;
     WriteBinaryData(SPosition, Position, SizeOf(Position));
    finally
      CloseKey;
    end;
  finally
    Free;
  end;
  if FFromApp and actStop.Enabled then
    actStop.Execute;
  Shell_NotifyIcon(NIM_DELETE, @FNotifyIconData);
end;

procedure TIBConnForm.lvConnectionsColumnClick(Sender: TObject;
  Column: TListColumn);
begin
  FAscending := not FAscending;
  if (FSortCol <> -1) and (FSortCol <> Column.Index) then
  begin
    FAscending := True;
    lvConnections.Columns[FSortCol].ImageIndex := -1;
  end;
  if FAscending then
    Column.ImageIndex := 1
  else
    Column.ImageIndex := 2;
  FSortCol := Column.Index;
  lvConnections.CustomSort(nil, FSortCol - 1);
end;

procedure TIBConnForm.lvConnectionsCompare(Sender: TObject; Item1,
  Item2: TListItem; Data: Integer; var Compare: Integer);
var
  SortFlag: Integer;
begin
  if FAscending then SortFlag := 1 else SortFlag := -1;
  if Data = -1 then
    Compare := SortFlag * AnsiCompareText(AlignIpAddress(Item1.Caption),
                                          AlignIpAddress(Item2.Caption))
  else
    Compare := SortFlag * AnsiCompareText(Item1.SubItems[Data],
                                          Item2.SubItems[Data]);
end;

procedure TIBConnForm.lvConnectionsKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key = 65) and (Shift = [ssCtrl]) then
    lvConnections.SelectAll;
end;

procedure TIBConnForm.actStartExecute(Sender: TObject);
begin
  IBConnThread := TIBConnThread.Create(False);
  IBSpyThread := TIBSpyThread.Create(False);
  SvrUdpThread := TSvrUdpThread.Create(False, UDP_PORT);
  actStart.Enabled := False;
  actStop.Enabled := not actStart.Enabled;
end;

procedure TIBConnForm.actStopExecute(Sender: TObject);
begin
  with IBConnThread do
  begin
    Terminate;
    if Suspended then
      Resume;
    WaitFor;
    Free;
  end;
  with IBSpyThread do
  begin
    FreeOnTerminate := True;
    Terminate;
  end;
  with SvrUdpThread do
  begin
    FreeOnTerminate := True;
    Terminate;
  end;
  actStop.Enabled := False;
  actStart.Enabled := not actStop.Enabled;
  if FFromApp then
  begin
    lvConnections.Items.Clear;
    UpdateStatus;
  end;
end;

procedure TIBConnForm.actOpenExecute(Sender: TObject);
begin
  if Visible then
    SetForegroundWindow(Handle)
  else
  begin
    Show;
    Forms.Application.BringToFront;
  end;
end;

procedure TIBConnForm.actExitExecute(Sender: TObject);
begin
  Forms.Application.Terminate;
end;

procedure TIBConnForm.actCloseConnectionExecute(Sender: TObject);
var
  i: Integer;
begin
  for i := lvConnections.Items.Count - 1 downto 0 do
    with lvConnections.Items[i] do
      if Selected then
      begin
        PMibTcpRow(Data)^.dwState := MIB_TCP_STATE_DELETE_TCB;
        SetTcpEntry(PMibTcpRow(Data));
      end;
end;

procedure TIBConnForm.actCloseConnectionUpdate(Sender: TObject);
begin
  (Sender as TAction).Enabled := lvConnections.SelCount > 0;
end;

procedure TIBConnForm.UpdateStatus;
begin
  StatusBar.SimpleText := Format(SCurrentConnections, [lvConnections.Items.Count]);
end;

procedure TIBConnForm.WMIBConnSvcIcon(var Message: TMessage);
var
  pt: TPoint;
begin
  case Message.LParam of
    WM_RBUTTONUP:
      begin
        SetForegroundWindow(Handle);
        GetCursorPos(pt);
        PopupMenu1.Popup(pt.x, pt.y);
      end;
    WM_LBUTTONDBLCLK:
      actOpen.Execute;
  end;
end;

procedure TIBConnForm.AddIcon;
begin
  with FNotifyIconData do
  begin
    cbSize := SizeOf(FNotifyIconData);
    Wnd := Handle;
    uID := $AE77;
    uFlags := NIF_MESSAGE or NIF_ICON or NIF_TIP;
    hIcon := Forms.Application.Icon.Handle;
    uCallbackMessage := WM_IBCONNSVCICON;
    StrCopy(szTip, PChar(Format(SIBConnSvcTip, [lvConnections.Items.Count])));
  end;
  Shell_NotifyIcon(NIM_ADD, @FNotifyIconData);
end;

procedure TIBConnForm.ModifyIcon;
var
  ResName: string;
begin
  if lvConnections.Items.Count = 0 then
    ResName := 'MAINICON'
  else
    ResName := 'MAINICON1';
  Forms.Application.Icon.Handle := LoadIcon(HInstance, PChar(ResName));
  with FNotifyIconData do
  begin
    hIcon := Forms.Application.Icon.Handle;
    StrCopy(szTip, PChar(Format(SIBConnSvcTip, [lvConnections.Items.Count])));
  end;
  Shell_NotifyIcon(NIM_MODIFY, @FNotifyIconData);
end;

procedure TIBConnForm.Initialize(FromApp: Boolean);
begin
  FFromApp := FromApp;
  if FromApp then
    actStart.Execute
  else
  begin
    N1.Visible := False;
    actStart.Visible := False;
    actStop.Visible := False;
    N2.Visible := False;
    actExit.Visible := False;
  end;
  AddIcon;
  FTaskMessage := RegisterWindowMessage('TaskbarCreated');
end;

{ TIBConnService }

procedure ServiceController(CtrlCode: DWord); stdcall;
begin
  IBConnService.Controller(CtrlCode);
end;

function TIBConnService.GetServiceController: TServiceController;
begin
  Result := ServiceController;
end;

constructor TIBConnService.CreateNew(AOwner: TComponent; Dummy: Integer);
begin
  inherited CreateNew(AOwner, Dummy);
  AllowPause := False;
  Interactive := True;
  DisplayName := SIBConnSvcTitle;
  Name := SServiceName;
  StartType := stManual;
  OnStart := Start;
  OnStop := Stop;
end;

procedure TIBConnService.Start(Sender: TService; var Started: Boolean);
begin
  IBConnForm.actStart.Execute;
  Started := True;
end;

procedure TIBConnService.Stop(Sender: TService; var Stopped: Boolean);
begin
  IBConnForm.actStop.Execute;
  PostMessage(IBConnForm.Handle, WM_QUIT, 0, 0);
  Stopped := True;
end;

{ TIBConnThread }

constructor TIBConnThread.Create(CreateSuspended: Boolean);
begin
  inherited Create(CreateSuspended);
  FIBConnTable := TList.Create;
  FUId := 0;
end;

destructor TIBConnThread.Destroy;
var
  P: Pointer;
begin
  with FIBConnTable do
  begin
    while Count > 0 do
    begin
      P := First;
      Remove(P);
      Dispose(P);
    end;
    Free;
  end;
  inherited Destroy;
end;

procedure TIBConnThread.Execute;
var
  TcpTable: PMibTcpTable;
  dwSize, dwResult: DWORD;
  i, j: Integer;
  NewConnect: Boolean;
  HostEnt: PHostEnt;
  FileHandle: Integer;
  LogStr: string;
begin
  while not Terminated do
  begin
    dwSize := 0;
    GetTcpTable(nil, @dwSize, False);
    if dwSize = 0 then Continue;
    TcpTable := AllocMem(dwSize);
    try
      dwResult := GetTcpTable(TcpTable, @dwSize, False);
      if dwResult <> NO_ERROR then Continue;
      with FIBConnTable do
      begin
        for i := 0 to Count - 1 do
          PIBConnRow(items[i])^.dwState := MIB_TCP_STATE_CLOSED;
        with TcpTable^ do
          for i := 0 to dwNumEntries - 1 do
            if (table[i].dwState = MIB_TCP_STATE_ESTAB) and
              (table[i].dwLocalPort = IBPort) then
            begin
              NewConnect := True;
              for j := 0 to Count - 1 do
                if (table[i].dwLocalAddr = PIBConnRow(items[j])^.dwLocalAddr) and
                  (table[i].dwLocalPort = PIBConnRow(items[j])^.dwLocalPort) and
                  (table[i].dwRemoteAddr = PIBConnRow(items[j])^.dwRemoteAddr) and
                  (table[i].dwRemotePort = PIBConnRow(items[j])^.dwRemotePort) then
                begin
                  NewConnect := False;
                  PIBConnRow(items[j])^.dwState := table[i].dwState;
                  Break;
                end;
              if NewConnect then
              begin
                New(FCurRow);
                CopyMemory(FCurRow, @table[i], SizeOf(TMibTcpRow));
                HostEnt := gethostbyaddr(@FCurRow^.dwRemoteAddr, 4, PF_INET);
                if Assigned(HostEnt) then
                  FCurRow^.HostName := HostEnt^.h_name
                else
                  FCurRow^.HostName := SHostUnknown;
                FCurRow^.dtConnect := Now;
                FCurRow^.dwUId := GetUId;
                if not SingleLine then
                begin
                  FileHandle := FileOpen(LogFile, fmOpenReadWrite or fmShareDenyNone);
                  if FileHandle <> -1 then
                  begin
                    LogStr := Format(SFmtStr, [ComputerName, SIBConnSvcTitle,
                                     FormatDateTime(SDTFmtStr, FCurRow^.dtConnect),
                                     SConnect, FormatIpAddress(FCurRow^.dwRemoteAddr),
                                     FCurRow^.HostName]);
                    FileSeek(FileHandle, 0, 2);
                    FileWrite(FileHandle, PChar(LogStr)^, Length(LogStr));
                    FileClose(FileHandle);
                  end;
                end;
                Synchronize(AddConnect);
              end;
            end;
        for i := Count - 1 downto 0 do
          if PIBConnRow(items[i])^.dwState = MIB_TCP_STATE_CLOSED then
          begin
            FCurRow := items[i];
            FCurRow^.dtDisconnect := Now;
            FileHandle := FileOpen(LogFile, fmOpenReadWrite or fmShareDenyNone);
            if FileHandle <> -1 then
            begin
              if SingleLine then
                LogStr := Format(SSingleFmtStr, [FormatIpAddress(FCurRow^.dwRemoteAddr),
                                 FCurRow^.HostName,
                                 FormatDateTime(SDTSingleFmtStr, FCurRow^.dtConnect),
                                 FormatDateTime(SDTSingleFmtStr, FCurRow^.dtDisconnect)])
              else
                LogStr := Format(SFmtStr, [ComputerName, SIBConnSvcTitle,
                                 FormatDateTime(SDTFmtStr, FCurRow^.dtDisconnect),
                                 SDisconnect, FormatIpAddress(FCurRow^.dwRemoteAddr),
                                 FCurRow^.HostName]);
              FileSeek(FileHandle, 0, 2);
              FileWrite(FileHandle, PChar(LogStr)^, Length(LogStr));
              FileClose(FileHandle);
            end;
            Synchronize(RemoveConnect);
          end;
      end;
    finally
      FreeMem(TcpTable);
    end;
    Sleep(TimeOut);
  end;
end;

procedure TIBConnThread.AddConnect;
begin
  FIBConnTable.Add(FCurRow);
  with IBConnForm, lvConnections do
  begin
    with Items.Add do
    begin
      ImageIndex := 0;
      Caption := FormatIpAddress(FCurRow^.dwRemoteAddr);
      SubItems.Add(FCurRow^.HostName);
      SubItems.Add(FormatDateTime(SDTFmtStr, FCurRow^.dtConnect));
      Data := FCurRow;
    end;
    if FSortCol <> - 1 then
      CustomSort(nil, FSortCol - 1);
    if not Assigned(ItemFocused) then
      Items[0].Focused := True;
    ModifyIcon;
    UpdateStatus;
  end;
end;

procedure TIBConnThread.RemoveConnect;
var
  Item: TListItem;
begin
  with IBConnForm do
  begin
    Item := lvConnections.FindData(0, FCurRow, True, False);
    if Assigned(Item) then
      Item.Delete;
    ModifyIcon;
    UpdateStatus;
  end;
  FIBConnTable.Remove(FCurRow);
  Dispose(FCurRow);
end;

function TIBConnThread.GetUId: DWORD;
begin
  Result := FUId;
  Inc(FUId);
  if FUId = High(DWORD) then
    FUId := 0;
end;

{ TIBSpyThread }

procedure TIBSpyThread.Execute;

  function FoundListener(dwPort: DWORD): Boolean;
  var
    Sock: TSocket;
    Addr: TSockAddrIn;
  begin
    Result := False;
    Sock := socket(PF_INET, SOCK_STREAM, IPPROTO_IP);
    if Sock <> INVALID_SOCKET then
    try
      Addr.sin_family := PF_INET;
      Addr.sin_port := dwPort;
      Addr.sin_addr.S_addr := INADDR_ANY;
      bind(Sock, Addr, SizeOf(Addr));
      if WSAGetLastError = WSAEADDRINUSE then
        Result := True;
    finally
      closesocket(Sock);
    end;
  end;

begin
  while not Terminated do
  begin
    if not FoundListener(IBPort) then
    begin
      if not IBConnThread.Suspended then
        IBConnThread.Suspend
    end
    else if IBConnThread.Suspended then
      IBConnThread.Resume;
    Sleep(SPY_TIMEOUT);
  end;
end;

{ TSvrUdpThread }

procedure TSvrUdpThread.Read;
var
  RecvLen, SentLen, TextLen, i: Integer;
  Buffer: Pointer;
  dwUId: DWORD;
begin
  RecvLen := GetRecvLength;
  GetMem(Buffer, RecvLen);
  try
    if RecvBuf(Buffer^, RecvLen) > 0 then
      case PDWORD(Buffer)^ of
        acBroadcast:
          begin
            PDWORD(Buffer)^ := acConfirm;
            SendBuf(Buffer^, AC_SIZE);
          end;
        acGetConInfo:
          with IBConnForm.lvConnections do
            if Items.Count > 0 then
              with TStringList.Create do
              try
                for i := 0 to Items.Count - 1 do
                  with Items[i] do
                    Add(Caption + cFldSep + SubItems[0] + cFldSep +
                        SubItems[1] + cFldSep + IntToStr(PIBConnRow(Data)^.dwUId));
                TextLen := Length(Text);
                SentLen := SendBuf(PChar(Text)^, TextLen);
                if SentLen = TextLen then
                begin
                  ReallocMem(Buffer, AC_SIZE * 2);
                  PDWORD(Buffer)^ := acConfirmSize;
                  PDWORD(DWORD(Buffer) + AC_SIZE)^ := SentLen;
                  SendBuf(Buffer^, AC_SIZE * 2);
                end;
              finally
                Free;
              end;
        acCloseConnect:
          begin
            dwUId := PDWORD(DWORD(Buffer) + AC_SIZE)^;
            with IBConnForm.lvConnections do
              for i := 0 to Items.Count - 1 do
                with Items[i] do
                  if PIBConnRow(Data)^.dwUId = dwUId then
                  begin
                    PMibTcpRow(Data)^.dwState := MIB_TCP_STATE_DELETE_TCB;
                    SetTcpEntry(Data);
                    Break;
                  end;
          end;
      end;
  finally
    FreeMem(Buffer);
  end;
end;

initialization
  Forms.Application.Title := SIBConnSvcTitle;
  SetThreadLocale($0409);
  GetFormatSettings;
  SetPriorityClass(GetCurrentProcess, IDLE_PRIORITY_CLASS);
end.
