unit UdpBase;

interface

uses
  Windows, SysUtils, Classes, WinSock;

const
  MAX_BUFFER_SIZE = 8192;

type

{ TCustomUdpThread class}

  TCustomUdpThread = class(TThread)
  private
    FSocket: TSocket;
    FAddrTo: TSockAddr;
    FAddrFrom: TSockAddr;
    FMaxBufferSize: Integer;
  protected
    procedure Execute; override;
    procedure Read; virtual;
  public
    constructor Create(CreateSuspended: Boolean; Port: Word;
      const IpHost: string = '');
    destructor Destroy; override;
    function GetRecvLength: Integer;
    function SendBuf(var Buf; Count: Integer): Integer;
    function RecvBuf(var Buf; Count: Integer): Integer;
    property SocketHandle: TSocket read FSocket;
    property AddrTo: TSockAddr read FAddrTo;
    property AddrFrom: TSockAddr read FAddrFrom;
    property MaxBufferSize: Integer read FMaxBufferSize write FMaxBufferSize
      default MAX_BUFFER_SIZE;
  end;

{ IP Utility functions}

function FormatIpAddress(Value: DWORD): string;
function AlignIpAddress(const IpAddress: string): string;

implementation

{ TCustomUdpThread }

constructor TCustomUdpThread.Create(CreateSuspended: Boolean; Port: Word;
  const IpHost: string = '');
var                                       
  OptVal: LongBool;
begin
  inherited Create(CreateSuspended);
  FSocket := socket(PF_INET, SOCK_DGRAM, IPPROTO_IP);
  if FSocket <> INVALID_SOCKET then
  begin
    FAddrTo.sin_family := AF_INET;
    FAddrTo.sin_port := htons(Port);
    if IpHost = '' then
      FAddrTo.sin_addr.S_addr := INADDR_ANY
    else
      FAddrTo.sin_addr.S_addr := inet_addr(PChar(IpHost));
    if FAddrTo.sin_addr.S_addr = INADDR_ANY then
      bind(FSocket, FAddrTo, SizeOf(FAddrTo))
    else if FAddrTo.sin_addr.S_addr = INADDR_BROADCAST then
    begin
      OptVal := True;
      setsockopt(FSocket, SOL_SOCKET, SO_BROADCAST, PChar(@OptVal), SizeOf(OptVal));
    end;
  end;
  FMaxBufferSize := MAX_BUFFER_SIZE;
end;

destructor TCustomUdpThread.Destroy;
begin
  if FSocket <> INVALID_SOCKET then
    closesocket(FSocket);
  inherited Destroy;
end;

procedure TCustomUdpThread.Execute;
var
  FDSet: TFDSet;
  TimeVal: TTimeVal;
begin
  while not Terminated and (FSocket <> INVALID_SOCKET) do
  begin
    FD_ZERO(FDSet);
    FD_SET(FSocket, FDSet);
    TimeVal.tv_sec := 1;
    TimeVal.tv_usec := 0;
    if (select(0, @FDSet, nil, nil, @TimeVal) > 0) and not Terminated then
      if GetRecvLength > 0 then
        Synchronize(Read);
  end;
end;

function TCustomUdpThread.GetRecvLength: Integer;
begin
  Result := 0;
  ioctlsocket(FSocket, FIONREAD, Result);
end;

function TCustomUdpThread.SendBuf(var Buf; Count: Integer): Integer;
var
  AddrTo: TSockAddr;
  ToLen, BufLen, SentLen: Integer;
begin
  Result := 0;
  AddrTo := FAddrTo;
  if FAddrTo.sin_addr.S_addr = INADDR_ANY then
    AddrTo := FAddrFrom;
  ToLen := SizeOf(AddrTo);
  if Count < FMaxBufferSize then
    Result := sendto(FSocket, Buf, Count, 0, AddrTo, ToLen)
  else
  begin
    SentLen := 0;
    while True do
    begin
      BufLen := FMaxBufferSize;
      if BufLen > Count - SentLen then
        BufLen := Count - SentLen;
      Result := sendto(FSocket, Pointer(DWORD(@Buf) + DWORD(SentLen))^,
                       BufLen, 0, AddrTo, ToLen);
      if Result = SOCKET_ERROR then
        Break;
      SentLen := SentLen + Result;
      if SentLen = Count then
      begin
        Result := SentLen;
        Break;
      end;
    end;
  end;
end;

function TCustomUdpThread.RecvBuf(var Buf; Count: Integer): Integer;
var
  FromLen: Integer;
begin
  FromLen := SizeOf(FAddrFrom);
  Result := recvfrom(FSocket, Buf, Count, 0, FAddrFrom, FromLen);
end;

procedure TCustomUdpThread.Read;
begin
end;

{ IP Utility }

function FormatIpAddress(Value: DWORD): string;
begin
  Result := Format('%d.%d.%d.%d', [Value and $FF,
                                  (Value shr 8) and $FF,
                                  (Value shr 16) and $FF,
                                  (Value shr 24) and $FF]);
end;

function AlignIpAddress(const IpAddress: string): string;
var
  P, Start: PChar;
  S: string;
begin
  Result := '';
  P := PChar(IpAddress);
  while P^ <> #0 do
  begin
    Start := P;
    while not (P^ in [#0, '.']) do Inc(P);
    SetString(S, Start, P - Start);
    Result := Result + Format('%3s', [S]);
    if P^ <> #0 then
    begin
      Result := Result + '.';
      Inc(P);
    end;
  end;
end;

var
  WSAData: TWSAData;

initialization
  WSAStartup($0101, WSAData);
finalization
  WSACleanup;
end.
