Unit IBase32;


interface

uses
  DB;

type
  EIBDatabaseError = EDatabaseError;

  ISC_STATUS = Longint;
  pISC_STATUS = ^ISC_STATUS;
  aISC_STATUS = array[0..19] of ISC_STATUS;

  USER_SEC_DATA = record
    sec_flags:     smallint;  // which fields are specified
    uid:           integer;   // the user's id
    gid:           integer;   // the user's group id
    protocol:      integer;   // protocol to use for connection
    server:        pchar;     // server to administer
    user_name:     pchar;     // the user's name
    password:      pchar;     // the user's password
    group_name:    pchar;     // the group name
    first_name:    pchar;     // the user's first name
    middle_name:   pchar;     // the user's middle name
    last_name:     pchar;     // the user's last name
    dba_user_name: pchar;     // the dba user name
    dba_password:  pchar;     // the dba password
  end;
  pUSER_SEC_DATA = ^USER_SEC_DATA;

{ User Name structure declarations }
  PServerInfo = ^TServerInfo;
  TServerInfo = record
    Name: string;
    Protocol: integer;
  end;

  TDBAInfo = record
    Name: string;
    Password: string;
  end;

  PUserInfo = ^TUserInfo;
  TUserInfo = record
    FirstName: string;
    MiddleName: string;
    LastName: string;
  end;



const
  IBaseDLL = 'gds32.dll';

  sec_protocol_tcpip      =      1;
  sec_protocol_netbeui    =      2;
  sec_protocol_spx        =      3;
  sec_protocol_local      =      4;

  sec_uid_spec		    = $0001;
  sec_gid_spec		    = $0002;
  sec_server_spec	    = $0004;
  sec_password_spec	    = $0008;
  sec_group_name_spec	    = $0010;
  sec_first_name_spec	    = $0020;
  sec_middle_name_spec      = $0040;
  sec_last_name_spec	    = $0080;
  sec_dba_user_name_spec    = $0100;
  sec_dba_password_spec     = $0200;


// Non-IB functions for error handling and user manipulations...
function ExtractServerName(DBString: string; Protocol: integer): string;

procedure IBCheck(var Status: aISC_STATUS);

function ModifyUser(ServerInfo: PServerInfo; DBAInfo: TDBAInfo;
                UserName, Password: string; UserInfo: PUserInfo): ISC_STATUS;

function AddUser(ServerInfo: PServerInfo; DBAInfo: TDBAInfo;
                UserName, Password: string; UserInfo: PUserInfo): ISC_STATUS;

function DeleteUser(ServerInfo: PServerInfo; DBAInfo: TDBAInfo;
                UserName: string): ISC_STATUS;


function isc_add_user(var status_vector: aISC_STATUS; var user_data: USER_SEC_DATA):
                ISC_STATUS ;stdcall;

function isc_delete_user(var status_vector: aISC_STATUS; var user_data: USER_SEC_DATA):
                ISC_STATUS ;stdcall;

function isc_modify_user(var status_vector: aISC_STATUS; var user_data: USER_SEC_DATA):
                ISC_STATUS ;stdcall;

function isc_interprete(status: PChar;
                var status_vector: pISC_STATUS): ISC_STATUS;stdcall;


implementation

uses SysUtils;

// remove the server name from the connect string depending on the protocol...
function ExtractServerName(DBString: string; Protocol: integer): string;
var
  Temp: string;

begin
  case Protocol of
    sec_protocol_tcpip:
      Result := Copy(DBString, 1, Pos(':', DBString) - 1);

    sec_protocol_netbeui:
    begin
      Temp := Copy(DBString, 3, Length(DBString) - 2);
      Result := Copy(Temp, 1, Pos('\', Temp) - 1);
    end;

    sec_protocol_spx:
      Result := Copy(DBString, 1, Pos('@', DBString) - 1);

    sec_protocol_local:
    begin
      Result := '';
    end;
  end;
end;

// throw an exception if an error occurs...
procedure IBCheck(var Status: aISC_STATUS);
var
  S, Total: string;
  P: pISC_STATUS;
  AllDone: ISC_STATUS;

begin
  if (Status[0] = 1) and (Status[1] <> 0) then
  begin
    P := @Status;
    // loop through the errors until isc_interprete returns 0...
    repeat
      SetLength(S, 512);
      alldone := isc_interprete(PChar(S), P);
      if alldone <> 0 then
      begin
        SetLength(S, StrLen(PChar(S)));
        Total := Total + S + #10#13;
      end;
    until alldone = 0;
    SetLength(Total, StrLen(Pchar(Total)) - 2);
    // raise an meaningful error...
    raise EIBDatabaseError.Create(Total);
  end;
end;

// modify an existing users information in the password database...
function ModifyUser(ServerInfo: PServerInfo; DBAInfo: TDBAInfo;
                UserName, Password: string; UserInfo: PUserInfo): ISC_STATUS;

var
  Status: aISC_STATUS;
  Sec: USER_SEC_DATA;
  Flags: smallint;

begin
  Flags := 0;
  FillChar(Sec, sizeof(Sec), 0);

  // if server information was given, add the informatino and set the flags...
  if ServerInfo <> nil then
  begin
    Sec.Server := PChar(ServerInfo^.Name);
    Sec.Protocol := ServerInfo^.Protocol;
    Flags := Flags or sec_server_spec;
  end
  else
    Sec.Protocol := sec_protocol_local;

  // add user and password informatino and set the flags...
  Sec.dba_user_name := PChar(DBAInfo.Name);
  Sec.dba_password := PChar(DBAInfo.Password);
  Flags := Flags or sec_dba_user_name_spec or sec_dba_password_spec;

  if UserInfo <> nil then
  begin
    Sec.first_name := PChar(UserInfo^.FirstName);
    Sec.middle_name := PChar(UserInfo^.MiddleName);
    Sec.last_name := PChar(UserInfo^.LastName);
    Flags := Flags or sec_first_name_spec or sec_middle_name_spec or sec_last_name_spec;
  end;
  Sec.user_name := PChar(UserName);
  Sec.password := PChar(Password);
  if Password <> '' then
    Flags := Flags or sec_password_spec;
  Sec.sec_flags := Flags;
  result := isc_modify_user(Status, Sec);
  IBCheck(Status);
end;

function AddUser(ServerInfo: PServerInfo; DBAInfo: TDBAInfo;
                  UserName, Password: string; UserInfo: PUserInfo): ISC_STATUS;

var
  Status: aISC_STATUS;
  Sec: USER_SEC_DATA;
  Flags: smallint;

begin
  Flags := 0;
  FillChar(Sec, sizeof(Sec), 0);

  if ServerInfo <> nil then
  begin
    Sec.Server := PChar(ServerInfo^.Name);
    Sec.Protocol := ServerInfo^.Protocol;
    Flags := Flags or sec_server_spec;
  end
  else
    Sec.Protocol := sec_protocol_local;

  Sec.dba_user_name := PChar(DBAInfo.Name);
  Sec.dba_password := PChar(DBAInfo.Password);
  Flags := Flags or sec_dba_user_name_spec or sec_dba_password_spec;

  if UserInfo <> nil then
  begin
    Sec.first_name := PChar(UserInfo^.FirstName);
    Sec.middle_name := PChar(UserInfo^.MiddleName);
    Sec.last_name := PChar(UserInfo^.LastName);
    Flags := Flags or sec_first_name_spec or sec_middle_name_spec or sec_last_name_spec;
  end;
  Sec.user_name := PChar(UserName);
  Sec.password := PChar(Password);
  Flags := Flags or sec_password_spec;
  Sec.sec_flags := Flags;
  result := isc_add_user(Status, Sec);
  IBCheck(Status);
end;

function DeleteUser(ServerInfo: PServerInfo; DBAInfo: tDBAInfo;
                  UserName: string): ISC_STATUS;
var
  Status: aISC_STATUS;
  Sec: USER_SEC_DATA;
  Flags: smallint;

begin
  Flags := 0;
  FillChar(Sec, sizeof(Sec), 0);

  if ServerInfo <> nil then
  begin
    Sec.Server := PChar(ServerInfo^.Name);
    Sec.Protocol := ServerInfo^.Protocol;
    Flags := Flags or sec_server_spec;
  end
  else
    Sec.Protocol := sec_protocol_local;

  Sec.dba_user_name := PChar(DBAInfo.Name);
  Sec.dba_password := PChar(DBAInfo.Password);
  Flags := Flags or sec_dba_user_name_spec or sec_dba_password_spec;

  Sec.user_name := PChar(UserName);
  Sec.sec_flags := Flags;
  result := isc_delete_user(Status, Sec);
  IBCheck(Status);
end;


function isc_add_user; external IBaseDLL;
function isc_modify_user; external IBaseDLL;
function isc_delete_user; external IBaseDLL;

function isc_interprete; external IBaseDLL;

end.

