library ListUdf;

uses
  SysUtils,
  Classes,
  ib_util in 'ib_util.pas',
  tb_consts in 'tb_consts.pas',
  tb_util in 'tb_util.pas';

{$R *.RES}


{      TDSC (  UDF  
 By descriptor)}

type TIBList = class(TList)
public
  destructor Destroy; override;
  function AddItem(v : PDSC) : boolean;
  function InsertItem(i : integer; v : PDSC): boolean;
  function SetItem(i : integer; v : PDSC): boolean;
  function GetItem(i : integer; v : PDSC) : boolean;
  function FindItem(v : PDSC) : integer;
  function SortItems : integer;
  function DeleteItem(i : integer) : integer;
  function ClearItems : integer;
end;

var IBLists : TThreadList;


{ TIbList }


{  , Null  }

function CmpDsc (Item1, Item2: Pointer): Integer;
var v1, v2 : PDSC;
begin
  Result := 0;
  v1 := Item1;
  v2 := Item2;
  if (isNull(v1) and IsNull(v2)) then
    Result := 0
  else if isNull(v1) then
    Result := -1
  else if isNull(v2) then
    Result := 1
  else if v1.dsc_dtype = v2.dsc_dtype then begin
    case v1.dsc_dtype of
      dtype_text, dtype_cstring, dtype_varying :
        Result := AnsiStrComp(PChar(get_string_value(v1)),PChar(get_string_value(v2)));
      dtype_real, dtype_double :
        if (get_double_value(v1) < get_double_value(v2)) then
          Result := -1
        else if (get_double_value(v1) > get_double_value(v2)) then
          Result := 1
        else
          Result := 0;
      dtype_short, dtype_long, dtype_int64 :
        if (get_int_value(v1) < get_int_value(v2)) then
          Result := -1
        else if (get_int_value(v1) > get_int_value(v2)) then
          Result := 1
        else
          Result := 0;
    end;
  end;
end;


{  ,   Null = Null}

function CmpDscFind (Item1, Item2: Pointer): Boolean;
var v1, v2 : PDSC;
begin
  v1 := Item1;
  v2 := Item2;
  Result := False;
  if (isNull(v1) and IsNull(v2)) then
    Result := True
  else if (isNull(v1)) or (isNull(v2)) then
    Result := False
  else if v1.dsc_dtype = v2.dsc_dtype then begin
    case v1.dsc_dtype of
      dtype_text, dtype_cstring, dtype_varying :
        Result := AnsiStrComp(PChar(get_string_value(v1)),PChar(get_string_value(v2))) = 0;
      dtype_real, dtype_double :
        Result := get_double_value(v1) = get_double_value(v2);
      dtype_short, dtype_long, dtype_int64 :
        Result := get_int_value(v1) = get_int_value(v2);
    end;
  end;
end;

function TIBList.AddItem(v : PDSC) : boolean;
begin
  try
    Add(CloneParam(v));
    Result := True;
  except
    Result := False;
  end;
end;

function TIBList.InsertItem(i : integer; v : PDSC): boolean;
begin
  try
    Insert(i, CloneParam(v));
    Result := True;
  except
    Result := False;
  end;
end;

function TIBList.GetItem(i: integer; v: PDSC): boolean;
begin
  Result := i < Count;
  if Result then
    try
      CopyParam(PDSC(Items[i]), v);
    except
      Result := False;
    end;
end;


function TIBList.SetItem(i : integer; v : PDSC): boolean;
begin
  Result := i < Count;
  if Result then
    try
      CopyParam(v, PDSC(Items[i]));
    except
      Result := False;
    end;
end;

function TIBList.DeleteItem(i: integer) : integer;
begin
  Result := -1;
  if i < Count then
  try
    if Assigned(Items[i]) then begin
      if (not IsNull(PDSC(Items[i]))) and (PDSC(Items[i])^.dsc_length > 0) and
         Assigned(PDSC(Items[i])^.dsc_address) then
        FreeMem(PDSC(Items[i])^.dsc_address);
      FreeMem(Items[i]);
    end;
    Delete(i);
    Result := 0;
  except
  end;
end;

function TIBList.FindItem(v: PDSC): integer;
begin
  Result := 0;
  while (Result < Count) and (not CmpDscFind(Items[Result],v)) do
    Inc(Result);
  if Result = Count then
    Result := -1;
end;

destructor TIBList.Destroy;
var i : integer;
begin
  for i := Pred(Count) downto 0 do
    try
      DeleteItem(i);
    except
    end;
  inherited;
end;

function TIBList.SortItems: integer;
var i : integer;
begin
  Result := 0;
  if Count > 0 then begin
    i := 1;
    while (i < Count) and (PDSC(Items[0])^.dsc_dtype = PDSC(Items[i])^.dsc_dtype) do
      Inc(i);
    if i = Count then
      Sort(CmpDsc)
    else
      Result := -1;
  end;
end;

function TIBList.ClearItems: integer;
var i : integer;
begin
  for i := Pred(Count) downto 0 do
    try
      DeleteItem(i);
    except
    end;
  Result := 0;
end;


function CreateIbList(var i : integer) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      Add(TIBList.Create);
      Result := Pred(Count);
    finally
      IBLists.UnlockList;
    end;
end;

function DeleteIbList(var i : integer) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) then begin
        TIBList(Items[i]).Destroy;
        Items[i] := nil;
        Result := 0;
      end;
    finally
      IBLists.UnlockList;
    end;
end;

function IbListAdd(var i : integer; v : PDSC) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) and TIBList(Items[i]).AddItem(v) then
        Result := 0;
    finally
      IBLists.UnlockList;
    end;
end;

function IbListSet(var i : integer; var j : integer; v : PDSC) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) and TIBList(Items[i]).SetItem(j, v) then
        Result := 0;
    finally
      IBLists.UnlockList;
    end;
end;

function IbListGet(var i : integer; var j : integer; v : PDSC) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) and TIBList(Items[i]).GetItem(j,v) then
        Result := 0;
    finally
      IBLists.UnlockList;
    end;
end;

function IbListInsert(var i : integer; var j : integer; v : PDSC) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) and TIBList(Items[i]).InsertItem(j,v) then
        Result := 0;
    finally
      IBLists.UnlockList;
    end;
end;


function IbListFind(var i : integer; v : PDSC) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) then
        Result := TIBList(Items[i]).FindItem(v);
    finally
      IBLists.UnlockList;
    end;
end;

function IbListSort(var i : integer) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) then
        Result := TIBList(Items[i]).SortItems;
    finally
      IBLists.UnlockList;
    end;
end;

function IbListClear(var i : integer) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) then begin
        Result := TIBList(Items[i]).ClearItems;
      end
    finally
      IBLists.UnlockList;
    end;
end;


function IbListDelete(var i : integer; var j : integer) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) then
        Result := TIBList(Items[i]).DeleteItem(j);
    finally
      IBLists.UnlockList;
    end;
end;

function IbListExchange(var i : integer; var j : integer; var k : integer) : integer; cdecl;
begin
  Result := -1;
  with IBLists.LockList do
    try
      if (i < Count) and Assigned(Items[i]) then begin
        TIBList(Items[i]).Exchange(j,k);
        Result := 0;
      end;
    finally
      IBLists.UnlockList;
    end;
end;


exports
  IbListAdd name 'IBLISTADD',
  IbListSet name 'IBLISTSET',
  IbListGet name 'IBLISTGET',
  IbListFind name 'IBLISTFIND',
  IbListSort name 'IBLISTSORT',
  IbListDelete name 'IBLISTDELETE',
  IbListClear name 'IBLISTCLEAR',
  IbListInsert name 'IBLISTINSERT',
  IbListExchange name 'IBLISTEXCHANGE',
  CreateIbList name 'CREATEIBLIST',
  DeleteIbList name 'DELETEIBLIST';


begin
  IsMultiThread := True;
  IBLists := TThreadList.Create;
end.
